/**
 * PazarRadar - Profit Calculator Engine
 *
 * Turk e-ticaret pazaryerleri (Trendyol, Hepsiburada, N11) icin
 * kar hesaplama motoru. Komisyon, KDV, platform ucreti ve kargo
 * dahil tum kesintileri hesaplayarak net kar analizi sunar.
 *
 * Bagimlilik: window.PazarRadar.CommissionDB (onceden yuklenmeli)
 *
 * @namespace window.PazarRadar.ProfitCalculator
 */
(function () {
  'use strict';

  window.PazarRadar = window.PazarRadar || {};

  // ─── Sabitler ───────────────────────────────────────────────

  /** Platform-spesifik sabit ucretler ve yapilandirma */
  const PLATFORM_CONFIG = Object.freeze({
    trendyol: Object.freeze({
      name: 'Trendyol',
      commissionBase: 'kdv_haric',
      platformFee: 8.49,
      defaultCargoPerDesi: 14.99,
      minCargo: 14.99
    }),
    hepsiburada: Object.freeze({
      name: 'Hepsiburada',
      commissionBase: 'kdv_dahil',
      platformFee: 10.00,
      defaultCargoPerDesi: 16.49,
      minCargo: 16.49
    }),
    n11: Object.freeze({
      name: 'N11',
      commissionBase: 'kdv_haric',
      platformFee: 7.00,
      defaultCargoPerDesi: 13.99,
      minCargo: 13.99
    })
  });

  /** Desteklenen platform isimleri */
  const VALID_PLATFORMS = Object.freeze(['trendyol', 'hepsiburada', 'n11']);

  /** Varsayilan KDV orani (%) */
  const DEFAULT_KDV_RATE = 20;

  /** Komisyon ve platform ucreti uzerindeki KDV orani (sabit %20) */
  const SERVICE_KDV_RATE = 0.20;

  /** Varsayilan desi agirlik */
  const DEFAULT_CARGO_WEIGHT = 1;

  // ─── Yardimci Fonksiyonlar ──────────────────────────────────

  /**
   * Sayiyi 2 ondalik basamaga yuvarlar.
   * @param {number} value - Yuvarlanacak sayi
   * @returns {number} 2 ondalik basamakli sayi
   */
  function round2(value) {
    return Math.round((value + Number.EPSILON) * 100) / 100;
  }

  /**
   * Verilen degerin gecerli pozitif sayi olup olmadigini kontrol eder.
   * @param {*} value - Kontrol edilecek deger
   * @returns {boolean}
   */
  function isValidPositiveNumber(value) {
    return typeof value === 'number' && Number.isFinite(value) && value > 0;
  }

  /**
   * Girdi parametrelerini dogrular ve varsayilan degerlerle birlestirerek normalize eder.
   * @param {Object} params - Ham girdi parametreleri
   * @returns {Object} Normalize edilmis parametreler
   * @throws {Error} Zorunlu alanlar eksikse veya gecersizse
   */
  function normalizeParams(params) {
    if (!params || typeof params !== 'object') {
      throw new Error('ProfitCalculator: Hesaplama parametreleri gereklidir.');
    }

    const platform = (typeof params.platform === 'string')
      ? params.platform.toLowerCase().trim()
      : '';

    if (!VALID_PLATFORMS.includes(platform)) {
      throw new Error(
        'ProfitCalculator: Gecersiz platform "' + params.platform + '". ' +
        'Gecerli platformlar: ' + VALID_PLATFORMS.join(', ')
      );
    }

    const salePrice = Number(params.salePrice);
    if (!isValidPositiveNumber(salePrice)) {
      throw new Error(
        'ProfitCalculator: Gecerli bir satis fiyati (salePrice) gereklidir. Alinan: ' + params.salePrice
      );
    }

    const costPrice = (params.costPrice != null && params.costPrice !== '')
      ? Number(params.costPrice)
      : null;

    if (costPrice !== null && (!Number.isFinite(costPrice) || costPrice < 0)) {
      throw new Error(
        'ProfitCalculator: Urun maliyeti (costPrice) gecerli bir sayi olmalidir. Alinan: ' + params.costPrice
      );
    }

    const cargoWeight = (params.cargoWeight != null && params.cargoWeight !== '')
      ? Number(params.cargoWeight)
      : DEFAULT_CARGO_WEIGHT;

    if (!Number.isFinite(cargoWeight) || cargoWeight < 0) {
      throw new Error(
        'ProfitCalculator: Kargo agirligi (cargoWeight) gecerli bir sayi olmalidir. Alinan: ' + params.cargoWeight
      );
    }

    const kdvRate = (params.kdvRate != null && params.kdvRate !== '')
      ? Number(params.kdvRate)
      : DEFAULT_KDV_RATE;

    if (!Number.isFinite(kdvRate) || kdvRate < 0 || kdvRate > 100) {
      throw new Error(
        'ProfitCalculator: KDV orani (kdvRate) 0-100 arasinda olmalidir. Alinan: ' + params.kdvRate
      );
    }

    const sellerPaysShipping = (params.sellerPaysShipping !== false);

    return Object.freeze({
      platform: platform,
      categoryKey: (typeof params.categoryKey === 'string') ? params.categoryKey.trim() : '',
      categoryName: (typeof params.categoryName === 'string') ? params.categoryName.trim() : '',
      salePrice: salePrice,
      costPrice: costPrice,
      cargoWeight: cargoWeight,
      kdvRate: kdvRate,
      sellerPaysShipping: sellerPaysShipping
    });
  }

  /**
   * CommissionDB'den komisyon oranini alir.
   * Oncelikle categoryKey ile arar, bulamazsa categoryName ile fuzzy match dener.
   * @param {string} platform - Platform adi
   * @param {string} categoryKey - Kategori anahtari
   * @param {string} categoryName - Kategori adi (fuzzy match icin)
   * @returns {number} Komisyon orani (%), bulunamazsa 0
   */
  function resolveCommissionRate(platform, categoryKey, categoryName) {
    const commissionDB = window.PazarRadar && window.PazarRadar.CommissionDB;

    if (!commissionDB) {
      // CommissionDB yuklu degilse varsayilan oran dondur
      return 0;
    }

    // Once categoryKey ile dene
    if (categoryKey && typeof commissionDB.getRate === 'function') {
      const rateResult = commissionDB.getRate(platform, categoryKey);
      if (rateResult && typeof rateResult === 'object' && Number.isFinite(rateResult.rate)) {
        // rate 0.085 gibi ondalik formatta, yuzdeye cevir (8.5)
        return round2(rateResult.rate * 100);
      }
      if (typeof rateResult === 'number' && Number.isFinite(rateResult)) {
        return rateResult;
      }
    }

    // categoryName ile fuzzy match dene (findCategory kullan)
    if (categoryName && typeof commissionDB.findCategory === 'function') {
      const found = commissionDB.findCategory(platform, categoryName);
      if (found && Number.isFinite(found.rate)) {
        return round2(found.rate * 100);
      }
    }

    // Hicbir sonuc yoksa varsayilan rate icin search dene
    if (typeof commissionDB.getDefaultRate === 'function') {
      const rate = commissionDB.getDefaultRate(platform);
      if (rate != null && Number.isFinite(rate)) {
        return rate;
      }
    }

    return 0;
  }

  /**
   * Tahmini kargo ucretini hesaplar.
   * @param {string} platform - Platform adi
   * @param {number} desiWeight - Desi agirligi
   * @param {boolean} sellerPaysShipping - Satici kargo oduyorsa true
   * @returns {number} Tahmini kargo ucreti (TL)
   */
  function estimateCargo(platform, desiWeight, sellerPaysShipping) {
    if (!sellerPaysShipping) {
      return 0;
    }

    const config = PLATFORM_CONFIG[platform];
    if (!config) {
      return 0;
    }

    const weight = Math.max(desiWeight, 1);
    const cargo = weight * config.defaultCargoPerDesi;

    return round2(Math.max(cargo, config.minCargo));
  }

  // ─── Ana Hesaplama Motoru ───────────────────────────────────

  /**
   * Tek bir platform icin tum kesintileri ve kar hesaplamasi yapar.
   * @param {Object} normalized - Normalize edilmis parametreler
   * @returns {Object} ProfitResult (frozen)
   */
  function computePlatformProfit(normalized) {
    const config = PLATFORM_CONFIG[normalized.platform];
    const commissionRate = resolveCommissionRate(
      normalized.platform,
      normalized.categoryKey,
      normalized.categoryName
    );

    // KDV hesaplamalari
    const salePrice = normalized.salePrice;
    const kdvRate = normalized.kdvRate;
    const kdvMultiplier = 1 + (kdvRate / 100);
    const salePriceExKDV = round2(salePrice / kdvMultiplier);
    const kdvAmount = round2(salePrice - salePriceExKDV);

    // Komisyon hesaplamalari - platforma gore KDV dahil/haric matrah
    let commissionAmount;
    if (config.commissionBase === 'kdv_dahil') {
      // Hepsiburada: KDV dahil fiyat uzerinden komisyon
      commissionAmount = round2(salePrice * commissionRate / 100);
    } else {
      // Trendyol, N11: KDV haric fiyat uzerinden komisyon
      commissionAmount = round2(salePriceExKDV * commissionRate / 100);
    }

    // Komisyon uzerindeki KDV (%20 sabit)
    const commissionKDV = round2(commissionAmount * SERVICE_KDV_RATE);

    // Platform hizmet bedeli
    const platformFee = config.platformFee;
    const platformFeeKDV = round2(platformFee * SERVICE_KDV_RATE);

    // Kargo ucreti tahmini
    const estimatedCargo = estimateCargo(
      normalized.platform,
      normalized.cargoWeight,
      normalized.sellerPaysShipping
    );

    // Toplam platform kesintileri
    const totalMarketplaceCosts = round2(
      commissionAmount + commissionKDV + platformFee + platformFeeKDV + estimatedCargo
    );

    // Net gelir
    const netRevenue = round2(salePrice - totalMarketplaceCosts);

    // Net kar (maliyet varsa)
    const costPrice = normalized.costPrice;
    const netProfit = (costPrice !== null) ? round2(netRevenue - costPrice) : null;

    // Net kar marji %
    let netMarginPercent = null;
    if (netProfit !== null && salePrice > 0) {
      netMarginPercent = round2((netProfit / salePrice) * 100);
    }

    // Detayli doekuem (UI icin)
    const breakdown = buildBreakdown({
      commissionRate: commissionRate,
      commissionAmount: commissionAmount,
      commissionKDV: commissionKDV,
      platformFee: platformFee,
      platformFeeKDV: platformFeeKDV,
      estimatedCargo: estimatedCargo,
      costPrice: costPrice,
      kdvAmount: kdvAmount
    });

    return Object.freeze({
      platform: normalized.platform,
      salePrice: round2(salePrice),
      salePriceExKDV: salePriceExKDV,
      kdvAmount: kdvAmount,
      kdvRate: kdvRate,

      commissionRate: commissionRate,
      commissionBase: config.commissionBase,
      commissionAmount: commissionAmount,
      commissionKDV: commissionKDV,

      platformFee: round2(platformFee),
      platformFeeKDV: platformFeeKDV,

      estimatedCargo: estimatedCargo,

      totalMarketplaceCosts: totalMarketplaceCosts,

      costPrice: costPrice,

      netRevenue: netRevenue,
      netProfit: netProfit,
      netMarginPercent: netMarginPercent,

      breakdown: Object.freeze(breakdown)
    });
  }

  /**
   * UI icin detayli kesinti doekuemue olusturur.
   * @param {Object} data - Hesaplanmis kesinti degerleri
   * @returns {Array<Object>} Breakdown satirlari
   */
  function buildBreakdown(data) {
    const lines = [];

    lines.push(Object.freeze({
      label: 'Komisyon (%' + data.commissionRate + ')',
      amount: round2(data.commissionAmount),
      type: 'fee'
    }));

    lines.push(Object.freeze({
      label: 'Komisyon KDV',
      amount: round2(data.commissionKDV),
      type: 'tax'
    }));

    lines.push(Object.freeze({
      label: 'Platform Hizmet Bedeli',
      amount: round2(data.platformFee),
      type: 'fee'
    }));

    lines.push(Object.freeze({
      label: 'Hizmet Bedeli KDV',
      amount: round2(data.platformFeeKDV),
      type: 'tax'
    }));

    if (data.estimatedCargo > 0) {
      lines.push(Object.freeze({
        label: 'Tahmini Kargo',
        amount: round2(data.estimatedCargo),
        type: 'fee'
      }));
    }

    if (data.costPrice !== null) {
      lines.push(Object.freeze({
        label: 'Urun Maliyeti',
        amount: round2(data.costPrice),
        type: 'cost'
      }));
    }

    return lines;
  }

  // ─── Public API ─────────────────────────────────────────────

  window.PazarRadar.ProfitCalculator = Object.freeze({

    /**
     * Tek bir platform icin kar hesaplamasi yapar.
     *
     * @param {Object} params - Hesaplama parametreleri
     * @param {string} params.platform - Platform adi ('trendyol' | 'hepsiburada' | 'n11')
     * @param {string} [params.categoryKey] - CommissionDB kategori anahtari
     * @param {string} [params.categoryName] - Kategori adi (fuzzy match icin)
     * @param {number} params.salePrice - KDV dahil satis fiyati (TL)
     * @param {number} [params.costPrice] - Urun maliyeti (TL, opsiyonel)
     * @param {number} [params.cargoWeight=1] - Desi agirligi
     * @param {number} [params.kdvRate=20] - KDV orani (%)
     * @param {boolean} [params.sellerPaysShipping=true] - Satici kargo oduyor mu
     * @returns {Object} ProfitResult - Detayli kar analizi (frozen object)
     * @throws {Error} Parametreler gecersizse
     *
     * @example
     * const result = PazarRadar.ProfitCalculator.calculate({
     *   platform: 'trendyol',
     *   categoryKey: 'elektronik_telefon',
     *   salePrice: 1000,
     *   costPrice: 600,
     *   cargoWeight: 1
     * });
     * // result.netProfit => kar tutari
     * // result.netMarginPercent => kar marji %
     */
    calculate: function calculate(params) {
      const normalized = normalizeParams(params);
      return computePlatformProfit(normalized);
    },

    /**
     * Ayni urun icin 3 platformu karsilastirarak en karli olanini belirler.
     *
     * @param {Object} params - Hesaplama parametreleri (platform alani kullanilmaz)
     * @param {string} [params.categoryKey] - CommissionDB kategori anahtari
     * @param {string} [params.categoryName] - Kategori adi (fuzzy match icin)
     * @param {number} params.salePrice - KDV dahil satis fiyati (TL)
     * @param {number} [params.costPrice] - Urun maliyeti (TL, opsiyonel)
     * @param {number} [params.cargoWeight=1] - Desi agirligi
     * @param {number} [params.kdvRate=20] - KDV orani (%)
     * @param {boolean} [params.sellerPaysShipping=true] - Satici kargo oduyor mu
     * @returns {Object} Karsilastirma sonucu (frozen object)
     * @returns {Object} return.trendyol - Trendyol ProfitResult
     * @returns {Object} return.hepsiburada - Hepsiburada ProfitResult
     * @returns {Object} return.n11 - N11 ProfitResult
     * @returns {string} return.best - En karli platform adi
     *
     * @example
     * const comparison = PazarRadar.ProfitCalculator.compareAll({
     *   categoryKey: 'giyim_kadin',
     *   salePrice: 500,
     *   costPrice: 200
     * });
     * // comparison.best => 'trendyol' (en karli platform)
     * // comparison.trendyol.netProfit => Trendyol net kari
     */
    compareAll: function compareAll(params) {
      const baseParams = params || {};

      const results = {};
      let bestPlatform = '';
      let bestNetRevenue = -Infinity;

      for (let i = 0; i < VALID_PLATFORMS.length; i++) {
        const platform = VALID_PLATFORMS[i];

        // Her platform icin ayri hesaplama
        const platformParams = {
          platform: platform,
          categoryKey: baseParams.categoryKey,
          categoryName: baseParams.categoryName,
          salePrice: baseParams.salePrice,
          costPrice: baseParams.costPrice,
          cargoWeight: baseParams.cargoWeight,
          kdvRate: baseParams.kdvRate,
          sellerPaysShipping: baseParams.sellerPaysShipping
        };

        try {
          const normalized = normalizeParams(platformParams);
          const result = computePlatformProfit(normalized);
          results[platform] = result;

          // En iyi platformu net gelire gore belirle
          if (result.netRevenue > bestNetRevenue) {
            bestNetRevenue = result.netRevenue;
            bestPlatform = platform;
          }
        } catch (error) {
          // Platform hesaplamasinda hata olursa null dondur
          results[platform] = null;
        }
      }

      return Object.freeze({
        trendyol: results.trendyol || null,
        hepsiburada: results.hepsiburada || null,
        n11: results.n11 || null,
        best: bestPlatform
      });
    },

    /**
     * Hizli kar marji hesaplamasi yapar. Sadece yuzde deger dondurur.
     *
     * @param {string} platform - Platform adi ('trendyol' | 'hepsiburada' | 'n11')
     * @param {string} categoryKey - CommissionDB kategori anahtari
     * @param {number} salePrice - KDV dahil satis fiyati (TL)
     * @param {number} costPrice - Urun maliyeti (TL)
     * @returns {number} Net kar marji yuzdesi, hesaplanamazsa 0
     *
     * @example
     * const margin = PazarRadar.ProfitCalculator.quickMargin(
     *   'trendyol', 'elektronik_telefon', 1000, 600
     * );
     * // margin => 15.23 (%)
     */
    quickMargin: function quickMargin(platform, categoryKey, salePrice, costPrice) {
      // Temel dogrulama
      if (!platform || !isValidPositiveNumber(Number(salePrice)) || !isValidPositiveNumber(Number(costPrice))) {
        return 0;
      }

      try {
        const result = computePlatformProfit(normalizeParams({
          platform: platform,
          categoryKey: categoryKey || '',
          categoryName: '',
          salePrice: Number(salePrice),
          costPrice: Number(costPrice),
          cargoWeight: DEFAULT_CARGO_WEIGHT,
          kdvRate: DEFAULT_KDV_RATE,
          sellerPaysShipping: true
        }));

        return (result.netMarginPercent !== null) ? result.netMarginPercent : 0;
      } catch (_error) {
        return 0;
      }
    }
  });
})();
