/**
 * PazarRadar - UI Injection Library
 *
 * Turk e-ticaret pazaryerleri (Trendyol, Hepsiburada, N11) uzerine
 * analiz widget'lari enjekte eden UI kutuphanesi. Shadow DOM ile
 * stil izolasyonu saglar, harici bagimlilik gerektirmez.
 *
 * Tum elemanlar 'data-pazarradar' attribute ile isaretlenir.
 * Temizlik icin cleanup() metodu ile toplu kaldirilabilir.
 *
 * Guvenlik notu: Tum icerikler DOM API'leri ile olusturulur,
 * innerHTML yerine textContent ve createElement tercih edilir.
 * Kullanici girdisi dogrudan DOM'a eklenmez.
 *
 * @namespace window.PazarRadar.InjectUI
 */
(function () {
  'use strict';

  window.PazarRadar = window.PazarRadar || {};

  // ─── Sabitler ───────────────────────────────────────────────

  /** Veri attribute'u - tum enjekte edilen elemanlar bunu tasir */
  var DATA_ATTR = 'data-pazarradar';

  /** CSS sinif on eki */
  var PREFIX = 'pr-';

  /** Overlay z-index degeri */
  var Z_INDEX = 999999;

  /** Animasyon suresi (ms) */
  var FADE_DURATION = 200;

  /** Toast bildirimi icin otomatik kapanma suresi (ms) */
  var TOAST_AUTO_CLOSE = 4000;

  /** Urun paneli genisligi (px) */
  var PANEL_WIDTH = 320;

  /** Aktif toast zamanlayicilari */
  var activeToasts = [];

  /** Takip edilen tum enjekte edilmis elemanlar */
  var injectedElements = [];

  // ─── Tasarim Tokenlari ──────────────────────────────────────

  /** Renk ve tipografi tokenlari */
  var TOKENS = Object.freeze({
    accent: '#f97316',
    accentLight: '#fbbf24',
    accentDark: '#ea580c',
    success: '#22c55e',
    danger: '#ef4444',
    warning: '#fbbf24',
    bg: 'rgba(15, 13, 10, 0.95)',
    bgGlass: 'rgba(255, 255, 255, 0.05)',
    border: 'rgba(255, 255, 255, 0.1)',
    text: '#f1f5f9',
    textSecondary: '#94a3b8',
    textMuted: '#64748b',
    font: "'Segoe UI', -apple-system, BlinkMacSystemFont, sans-serif"
  });

  // ─── CSS Degiskenleri ───────────────────────────────────────

  /**
   * Tum widget'larda kullanilan ortak CSS degiskenleri
   * Shadow DOM icindeki :host seviyesinde tanimlanan temel stiller
   */
  var CSS_VARIABLES = [
    '--pr-accent: ' + TOKENS.accent,
    '--pr-accent-light: ' + TOKENS.accentLight,
    '--pr-accent-dark: ' + TOKENS.accentDark,
    '--pr-success: ' + TOKENS.success,
    '--pr-danger: ' + TOKENS.danger,
    '--pr-warning: ' + TOKENS.warning,
    '--pr-bg: ' + TOKENS.bg,
    '--pr-bg-glass: ' + TOKENS.bgGlass,
    '--pr-border: ' + TOKENS.border,
    '--pr-text: ' + TOKENS.text,
    '--pr-text-secondary: ' + TOKENS.textSecondary,
    '--pr-text-muted: ' + TOKENS.textMuted,
    '--pr-font: ' + TOKENS.font
  ].join('; ');

  // ─── Yardimci Fonksiyonlar ──────────────────────────────────

  /**
   * Sayiyi Turkce formatta bicimlendirir (binlik: nokta, ondalik: virgul).
   * @param {number} value - Bicimlendirilecek sayi
   * @param {number} [decimals=0] - Ondalik basamak sayisi
   * @returns {string} Formatlanmis sayi
   */
  function formatNumber(value, decimals) {
    if (value == null || !Number.isFinite(value)) {
      return '-';
    }
    var dec = (typeof decimals === 'number') ? decimals : 0;
    var parts = value.toFixed(dec).split('.');
    parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, '.');
    return parts.join(',');
  }

  /**
   * Buyuk sayilari kisaltilmis formatta gosterir (134K, 1,2M gibi).
   * @param {number} value - Kisaltilacak sayi
   * @returns {string} Kisaltilmis format
   */
  function formatCompact(value) {
    if (value == null || !Number.isFinite(value)) {
      return '-';
    }
    if (Math.abs(value) >= 1000000) {
      return formatNumber(value / 1000000, 1) + 'M';
    }
    if (Math.abs(value) >= 1000) {
      return formatNumber(value / 1000, 0) + 'K';
    }
    return formatNumber(value, 0);
  }

  /**
   * Para birimi olarak TL formatinda gosterir.
   * @param {number} value - Fiyat
   * @param {number} [decimals=2] - Ondalik basamak
   * @returns {string} Formatlanmis fiyat
   */
  function formatCurrency(value, decimals) {
    var dec = (typeof decimals === 'number') ? decimals : 2;
    return formatNumber(value, dec) + ' TL';
  }

  /**
   * Yuzde degerini formatlar.
   * @param {number} value - Yuzde degeri
   * @returns {string} %xx.x formatinda
   */
  function formatPercent(value) {
    if (value == null || !Number.isFinite(value)) {
      return '-%';
    }
    return '%' + formatNumber(value, 1);
  }

  /**
   * Tarihi kisa Turkce formatta gosterir (15 Oca, 03 Sub gibi).
   * @param {Date|string|number} date - Tarih
   * @returns {string}
   */
  function formatDateShort(date) {
    var months = ['Oca', 'Sub', 'Mar', 'Nis', 'May', 'Haz',
      'Tem', 'Agu', 'Eyl', 'Eki', 'Kas', 'Ara'];
    var d = (date instanceof Date) ? date : new Date(date);
    if (isNaN(d.getTime())) {
      return '-';
    }
    var day = String(d.getDate()).padStart(2, '0');
    return day + ' ' + months[d.getMonth()];
  }

  /**
   * Belirtilen marja gore renk belirler (yesil/kirmizi skala).
   * @param {number} value - Marj degeri
   * @param {number} [goodThreshold=15] - "iyi" esik degeri
   * @param {number} [badThreshold=5] - "kotu" esik degeri
   * @returns {string} Hex renk kodu
   */
  function getMarginColor(value, goodThreshold, badThreshold) {
    var good = (typeof goodThreshold === 'number') ? goodThreshold : 15;
    var bad = (typeof badThreshold === 'number') ? badThreshold : 5;
    if (!Number.isFinite(value)) {
      return TOKENS.textMuted;
    }
    if (value >= good) {
      return TOKENS.success;
    }
    if (value <= bad) {
      return TOKENS.danger;
    }
    return TOKENS.warning;
  }

  /**
   * Niche skor degerine gore renk belirler.
   * @param {number} score - Skor (0-10)
   * @returns {string} Hex renk kodu
   */
  function getScoreColor(score) {
    if (!Number.isFinite(score)) {
      return TOKENS.textMuted;
    }
    if (score >= 3.5) {
      return TOKENS.success;
    }
    if (score >= 2) {
      return TOKENS.warning;
    }
    return TOKENS.danger;
  }

  /**
   * Trend yonu icin ikon ve metin olusturur.
   * @param {string} trend - 'up', 'down', 'stable'
   * @returns {{icon: string, label: string, color: string}}
   */
  function getTrendInfo(trend) {
    var trendMap = {
      up: { icon: '\u2191', label: 'Yukseliyor', color: TOKENS.success },
      down: { icon: '\u2193', label: 'Dusuyor', color: TOKENS.danger },
      stable: { icon: '\u2194', label: 'Stabil', color: TOKENS.warning }
    };
    return trendMap[trend] || trendMap.stable;
  }

  /**
   * Benzersiz eleman kimligi uretir.
   * @param {string} widgetType - Widget tipi
   * @returns {string}
   */
  function generateId(widgetType) {
    return PREFIX + widgetType + '-' + Date.now() + '-' + Math.random().toString(36).substring(2, 8);
  }

  // ─── Guvenli DOM Yardimcilari ──────────────────────────────

  /**
   * Guvenli bir sekilde HTML elemani olusturur.
   * Hicbir zaman innerHTML kullanmaz, tum icerik textContent ile eklenir.
   *
   * @param {string} tag - HTML element tagi
   * @param {Object} [attrs] - Attribute'lar (className, style, vb.)
   * @param {string} [text] - textContent olarak eklenecek metin
   * @param {Array<HTMLElement>} [children] - Alt elemanlar
   * @returns {HTMLElement}
   */
  function el(tag, attrs, text, children) {
    var elem = document.createElement(tag);

    if (attrs) {
      if (attrs.className) {
        elem.className = attrs.className;
      }
      if (attrs.style) {
        elem.style.cssText = attrs.style;
      }
      if (attrs.id) {
        elem.id = attrs.id;
      }
      if (attrs.title) {
        elem.title = attrs.title;
      }
      if (attrs.type) {
        elem.type = attrs.type;
      }
      if (attrs.placeholder) {
        elem.placeholder = attrs.placeholder;
      }
      if (attrs.min) {
        elem.min = attrs.min;
      }
      if (attrs.step) {
        elem.step = attrs.step;
      }
      if (attrs.ariaLabel) {
        elem.setAttribute('aria-label', attrs.ariaLabel);
      }
    }

    if (text != null) {
      elem.textContent = text;
    }

    if (children) {
      for (var i = 0; i < children.length; i++) {
        if (children[i]) {
          elem.appendChild(children[i]);
        }
      }
    }

    return elem;
  }

  /**
   * SVG elemani olusturur (SVG namespace ile).
   * @param {string} tag - SVG element tagi
   * @param {Object} [attrs] - SVG attribute'lari
   * @returns {SVGElement}
   */
  function svgEl(tag, attrs) {
    var ns = 'http://www.w3.org/2000/svg';
    var elem = document.createElementNS(ns, tag);
    if (attrs) {
      var keys = Object.keys(attrs);
      for (var i = 0; i < keys.length; i++) {
        elem.setAttribute(keys[i], attrs[keys[i]]);
      }
    }
    return elem;
  }

  /**
   * Shadow DOM iceren bir host element olusturur.
   * @param {string} widgetType - Widget tipi (CSS sinifinda kullanilir)
   * @param {string} css - Uygulanacak CSS metni
   * @returns {{host: HTMLElement, shadow: ShadowRoot}}
   */
  function createShadowHost(widgetType, css) {
    var host = document.createElement('div');
    host.setAttribute(DATA_ATTR, widgetType);
    host.id = generateId(widgetType);
    host.style.cssText = 'all: initial; font-family: ' + TOKENS.font + ';';

    var shadow = host.attachShadow({ mode: 'closed' });

    var style = document.createElement('style');
    style.textContent = ':host { ' + CSS_VARIABLES + '; } ' + getBaseCSS() + ' ' + (css || '');
    shadow.appendChild(style);

    injectedElements.push(host);
    return { host: host, shadow: shadow };
  }

  /**
   * Basit (Shadow DOM'suz) bir konteyner element olusturur.
   * @param {string} widgetType - Widget tipi
   * @param {string} [tagName='div'] - HTML element tagi
   * @returns {HTMLElement}
   */
  function createSimpleElement(widgetType, tagName) {
    var elem = document.createElement(tagName || 'div');
    elem.setAttribute(DATA_ATTR, widgetType);
    elem.id = generateId(widgetType);
    injectedElements.push(elem);
    return elem;
  }

  /**
   * Elemente fade-in animasyonu uygular.
   * @param {HTMLElement} element
   */
  function fadeIn(element) {
    element.style.opacity = '0';
    element.style.transition = 'opacity ' + FADE_DURATION + 'ms ease-in-out';
    // Frame bekleme - tarayicinin stili hesaplamasina izin ver
    requestAnimationFrame(function () {
      requestAnimationFrame(function () {
        element.style.opacity = '1';
      });
    });
  }

  /**
   * Elemente fade-out animasyonu uygulayip DOM'dan kaldirir.
   * @param {HTMLElement} element
   * @param {Function} [callback] - Kaldirilma sonrasi cagrilacak fonksiyon
   */
  function fadeOut(element, callback) {
    element.style.transition = 'opacity ' + FADE_DURATION + 'ms ease-in-out';
    element.style.opacity = '0';
    setTimeout(function () {
      if (element.parentNode) {
        element.parentNode.removeChild(element);
      }
      if (typeof callback === 'function') {
        callback();
      }
    }, FADE_DURATION);
  }

  /**
   * Elemani surukleme (drag) islevi ile donatan yardimci fonksiyon.
   * @param {HTMLElement} dragHandle - Suruklemeyi baslatan alan
   * @param {HTMLElement} moveTarget - Hareket ettirilecek eleman
   */
  function makeDraggable(dragHandle, moveTarget) {
    var isDragging = false;
    var offsetX = 0;
    var offsetY = 0;

    function onMouseDown(e) {
      if (e.button !== 0) {
        return;
      }
      isDragging = true;
      var rect = moveTarget.getBoundingClientRect();
      offsetX = e.clientX - rect.left;
      offsetY = e.clientY - rect.top;
      dragHandle.style.cursor = 'grabbing';
      e.preventDefault();
    }

    function onMouseMove(e) {
      if (!isDragging) {
        return;
      }
      var newLeft = e.clientX - offsetX;
      var newTop = e.clientY - offsetY;

      // Ekran sinirlarini kontrol et
      var maxX = window.innerWidth - moveTarget.offsetWidth;
      var maxY = window.innerHeight - moveTarget.offsetHeight;
      newLeft = Math.max(0, Math.min(newLeft, maxX));
      newTop = Math.max(0, Math.min(newTop, maxY));

      moveTarget.style.left = newLeft + 'px';
      moveTarget.style.top = newTop + 'px';
      moveTarget.style.right = 'auto';
    }

    function onMouseUp() {
      if (!isDragging) {
        return;
      }
      isDragging = false;
      dragHandle.style.cursor = 'grab';
    }

    dragHandle.style.cursor = 'grab';
    dragHandle.addEventListener('mousedown', onMouseDown);
    document.addEventListener('mousemove', onMouseMove);
    document.addEventListener('mouseup', onMouseUp);
  }

  /**
   * Badge logosunu DOM elemani olarak olusturur (guvenli).
   * @returns {HTMLElement}
   */
  function createBadgeLogo() {
    var logo = el('span', { className: 'pr-badge-logo' });
    logo.appendChild(document.createTextNode('\uD83D\uDCCA '));
    logo.appendChild(document.createTextNode('PazarRadar'));
    return logo;
  }

  /**
   * Badge logosunu belirli bir metinle DOM elemani olarak olusturur.
   * @param {string} labelText - Logo yanindaki metin
   * @returns {HTMLElement}
   */
  function createBadgeLogoWithText(labelText) {
    var logo = el('span', { className: 'pr-badge-logo' });
    logo.appendChild(document.createTextNode('\uD83D\uDCCA '));
    logo.appendChild(document.createTextNode(labelText));
    return logo;
  }

  // ─── Temel CSS ──────────────────────────────────────────────

  /**
   * Tum Shadow DOM widget'larinda ortak olan temel stiller.
   * @returns {string} CSS metni
   */
  function getBaseCSS() {
    return [
      '*, *::before, *::after {',
      '  box-sizing: border-box;',
      '  margin: 0;',
      '  padding: 0;',
      '}',
      '.pr-container {',
      '  font-family: var(--pr-font);',
      '  color: var(--pr-text);',
      '  font-size: 13px;',
      '  line-height: 1.5;',
      '  -webkit-font-smoothing: antialiased;',
      '  -moz-osx-font-smoothing: grayscale;',
      '}',
      '.pr-glass {',
      '  background: var(--pr-bg);',
      '  backdrop-filter: blur(12px);',
      '  -webkit-backdrop-filter: blur(12px);',
      '  border: 1px solid var(--pr-border);',
      '  border-radius: 8px;',
      '}',
      '.pr-glass-light {',
      '  background: var(--pr-bg-glass);',
      '  border: 1px solid var(--pr-border);',
      '  border-radius: 6px;',
      '}',
      '.pr-accent-text { color: var(--pr-accent); }',
      '.pr-success-text { color: var(--pr-success); }',
      '.pr-danger-text { color: var(--pr-danger); }',
      '.pr-warning-text { color: var(--pr-warning); }',
      '.pr-muted-text { color: var(--pr-text-muted); }',
      '.pr-secondary-text { color: var(--pr-text-secondary); }',
      '.pr-row {',
      '  display: flex;',
      '  align-items: center;',
      '  gap: 8px;',
      '}',
      '.pr-row-between {',
      '  display: flex;',
      '  align-items: center;',
      '  justify-content: space-between;',
      '}',
      '.pr-divider {',
      '  border: none;',
      '  border-top: 1px solid var(--pr-border);',
      '  margin: 8px 0;',
      '}',
      '.pr-btn {',
      '  display: inline-flex;',
      '  align-items: center;',
      '  justify-content: center;',
      '  gap: 6px;',
      '  padding: 8px 16px;',
      '  border-radius: 6px;',
      '  border: none;',
      '  cursor: pointer;',
      '  font-family: var(--pr-font);',
      '  font-size: 13px;',
      '  font-weight: 600;',
      '  transition: background-color 150ms ease, transform 100ms ease;',
      '}',
      '.pr-btn:active { transform: scale(0.97); }',
      '.pr-btn-accent {',
      '  background: var(--pr-accent);',
      '  color: #fff;',
      '}',
      '.pr-btn-accent:hover { background: var(--pr-accent-dark); }',
      '.pr-btn-ghost {',
      '  background: transparent;',
      '  color: var(--pr-text-secondary);',
      '  border: 1px solid var(--pr-border);',
      '}',
      '.pr-btn-ghost:hover {',
      '  background: var(--pr-bg-glass);',
      '  color: var(--pr-text);',
      '}',
      '.pr-close-btn {',
      '  background: none;',
      '  border: none;',
      '  color: var(--pr-text-muted);',
      '  cursor: pointer;',
      '  padding: 4px;',
      '  font-size: 18px;',
      '  line-height: 1;',
      '  transition: color 150ms ease;',
      '}',
      '.pr-close-btn:hover { color: var(--pr-text); }',
      '.pr-section-title {',
      '  font-size: 11px;',
      '  font-weight: 700;',
      '  text-transform: uppercase;',
      '  letter-spacing: 0.5px;',
      '  color: var(--pr-text-muted);',
      '  padding: 6px 0 4px 0;',
      '}',
      '.pr-input {',
      '  background: rgba(255,255,255,0.07);',
      '  border: 1px solid var(--pr-border);',
      '  border-radius: 4px;',
      '  color: var(--pr-text);',
      '  font-family: var(--pr-font);',
      '  font-size: 13px;',
      '  padding: 6px 8px;',
      '  outline: none;',
      '  width: 100%;',
      '  transition: border-color 150ms ease;',
      '}',
      '.pr-input:focus { border-color: var(--pr-accent); }',
      '.pr-input::placeholder { color: var(--pr-text-muted); }',
      '.pr-badge-logo {',
      '  font-size: 12px;',
      '  font-weight: 700;',
      '  color: var(--pr-accent);',
      '  display: flex;',
      '  align-items: center;',
      '  gap: 4px;',
      '}',
      '.pr-collapsible-header {',
      '  cursor: pointer;',
      '  user-select: none;',
      '  display: flex;',
      '  align-items: center;',
      '  justify-content: space-between;',
      '}',
      '.pr-collapsible-header .pr-chevron {',
      '  transition: transform 200ms ease;',
      '  font-size: 10px;',
      '  color: var(--pr-text-muted);',
      '}',
      '.pr-collapsible-header.pr-collapsed .pr-chevron {',
      '  transform: rotate(-90deg);',
      '}'
    ].join('\n');
  }

  // ─── Yardimci UI Builder Fonksiyonlari ─────────────────────

  /**
   * Panel icinde kullanilan veri satiri olusturur.
   * @param {string} label - Satir etiketi
   * @param {string} value - Satir degeri
   * @param {string} [color] - Deger rengi
   * @returns {HTMLElement}
   */
  function buildDataRow(label, value, color) {
    var labelSpan = el('span', { className: 'pr-data-label' }, label);
    var valueSpan = el('span', {
      className: 'pr-data-value',
      style: color ? 'color:' + color : ''
    }, value);
    return el('div', { className: 'pr-data-row' }, null, [labelSpan, valueSpan]);
  }

  /**
   * FREE kullanicilar icin value'su bulanik veri satiri olusturur.
   * isPro true ise normal gosterir, false ise blur uygular.
   * @param {string} label
   * @param {string} value
   * @param {boolean} isPro
   * @param {string} [color]
   * @returns {HTMLElement}
   */
  function buildBlurredDataRow(label, value, isPro, color) {
    if (isPro) {
      return buildDataRow(label, value, color);
    }
    var labelSpan = el('span', { className: 'pr-data-label' }, label);
    var valueSpan = el('span', {
      className: 'pr-data-value pr-blur-value',
      style: color ? 'color:' + color : ''
    }, value);
    return el('div', { className: 'pr-data-row' }, null, [labelSpan, valueSpan]);
  }

  /**
   * Yatay ayirici cizgi olusturur.
   * @returns {HTMLElement}
   */
  function createDivider() {
    return el('hr', { className: 'pr-divider' });
  }

  /**
   * Kesikli ayirici cizgi olusturur.
   * @returns {HTMLElement}
   */
  function buildSeparator() {
    return el('hr', { className: 'pr-data-separator' });
  }

  /**
   * Kapatma butonu olusturur.
   * @param {Function} onClick - Tiklanma isleyicisi
   * @returns {HTMLElement}
   */
  function buildCloseButton(onClick) {
    var btn = el('button', { className: 'pr-close-btn', title: 'Kapat' }, '\u2715');
    btn.addEventListener('click', onClick);
    return btn;
  }

  /**
   * Urun skoru bolumunu olusturur (ilerleme cubugu ile).
   * @param {number} score - Skor degeri (0-10)
   * @returns {HTMLElement}
   */
  function buildScoreSection(score) {
    var safeScore = Number(score) || 0;
    var color = getScoreColor(safeScore);
    var percent = Math.max(0, Math.min(safeScore / 5, 1)) * 100;

    var section = el('div');

    // Baslik
    var title = el('div', {
      style: 'font-size:11px; font-weight:700; text-transform:uppercase; ' +
        'letter-spacing:0.5px; color:' + TOKENS.textMuted + '; margin-bottom:6px;'
    }, 'URUN SKORU');

    // Alt aciklama
    var subtitle = el('div', {
      style: 'font-size:9px; color:' + TOKENS.textMuted + '; margin-bottom:6px; opacity:0.7;'
    }, 'Puan \u00B7 Yorum \u00B7 Talep \u00B7 Trend \u00B7 Deger');

    // Skor cubugu
    var scoreBar = el('div', { className: 'pr-score-bar' });

    // Skor degeri
    var scoreValueWrap = el('div');
    var scoreNum = el('span', { className: 'pr-score-value', style: 'color:' + color }, safeScore.toFixed(1));
    var scoreMax = el('span', { className: 'pr-score-max' }, '/5');
    scoreValueWrap.appendChild(scoreNum);
    scoreValueWrap.appendChild(scoreMax);

    // Ilerleme cubugu
    var track = el('div', { className: 'pr-score-track' });
    var fill = el('div', {
      className: 'pr-score-fill',
      style: 'width:' + percent + '%; background:' + color
    });
    track.appendChild(fill);

    scoreBar.appendChild(scoreValueWrap);
    scoreBar.appendChild(track);

    section.appendChild(title);
    section.appendChild(subtitle);
    section.appendChild(scoreBar);
    return section;
  }

  /**
   * Acilip kapanabilir bolum olusturur.
   * @param {string} title - Bolum basligi
   * @param {Function} buildContent - Icerik olusturucu (content div parametre olarak verilir)
   * @returns {HTMLElement}
   */
  function buildCollapsibleSection(title, buildContent) {
    var section = el('div', { className: 'pr-section' });

    var header = el('div', { className: 'pr-section-header pr-section-title' });
    var titleSpan = el('span', null, title);
    var chevron = el('span', { className: 'pr-section-chevron' }, '\u25BC');
    header.appendChild(titleSpan);
    header.appendChild(chevron);

    var content = el('div', {
      className: 'pr-section-content',
      style: 'max-height: 500px; overflow: hidden; transition: max-height 200ms ease;'
    });

    if (typeof buildContent === 'function') {
      buildContent(content);
    }

    var isCollapsed = false;
    header.addEventListener('click', function () {
      isCollapsed = !isCollapsed;
      header.classList.toggle('pr-collapsed', isCollapsed);
      if (isCollapsed) {
        content.style.maxHeight = '0';
        content.style.overflow = 'hidden';
      } else {
        content.style.maxHeight = content.scrollHeight + 'px';
        content.style.overflow = 'hidden';
      }
    });

    section.appendChild(header);
    section.appendChild(content);
    return section;
  }

  /**
   * Dikey ayirici cizgi (stat bar icin).
   * @returns {HTMLElement}
   */
  function buildStatDivider() {
    return el('span', {
      className: 'pr-agg-stat-divider',
      style: 'width:1px; height:16px; background:' + TOKENS.border + '; display:inline-block;'
    });
  }

  // ─── 1. Floating Badge ─────────────────────────────────────

  /**
   * Tum sayfalarda gorunen PazarRadar yuvarlak floating badge olusturur.
   * Tiklandiginda uygulamanin ana panelini acar.
   * @returns {HTMLElement} Badge host elemani
   */
  function createBadge() {
    var css = [
      '.pr-badge-float {',
      '  position: fixed;',
      '  bottom: 20px;',
      '  right: 20px;',
      '  width: 48px;',
      '  height: 48px;',
      '  border-radius: 50%;',
      '  background: linear-gradient(135deg, ' + TOKENS.accent + ', ' + TOKENS.accentDark + ');',
      '  display: flex;',
      '  align-items: center;',
      '  justify-content: center;',
      '  cursor: pointer;',
      '  z-index: ' + Z_INDEX + ';',
      '  box-shadow: 0 4px 16px rgba(249, 115, 22, 0.35);',
      '  transition: transform 200ms ease, box-shadow 200ms ease;',
      '}',
      '.pr-badge-float:hover {',
      '  transform: scale(1.08);',
      '  box-shadow: 0 6px 24px rgba(249, 115, 22, 0.5);',
      '}',
      '.pr-badge-float:active {',
      '  transform: scale(0.95);',
      '}',
      '.pr-badge-icon {',
      '  color: #fff;',
      '  font-size: 22px;',
      '  font-weight: 700;',
      '  line-height: 1;',
      '  pointer-events: none;',
      '  display: flex;',
      '  align-items: center;',
      '  justify-content: center;',
      '}'
    ].join('\n');

    var parts = createShadowHost('badge', css);
    var badge = el('div', { className: 'pr-badge-float' });

    // SVG grafik ikonu (guvenli: programatik olusturulmus)
    var iconWrap = el('span', { className: 'pr-badge-icon' });
    var svg = svgEl('svg', {
      width: '22',
      height: '22',
      viewBox: '0 0 24 24',
      fill: 'none',
      stroke: 'currentColor',
      'stroke-width': '2.5',
      'stroke-linecap': 'round',
      'stroke-linejoin': 'round'
    });
    var path1 = svgEl('path', { d: 'M3 3v18h18' });
    var path2 = svgEl('path', { d: 'M18.7 8l-5.1 5.2-2.8-2.7L7 14.3' });
    svg.appendChild(path1);
    svg.appendChild(path2);
    iconWrap.appendChild(svg);
    badge.appendChild(iconWrap);

    // Tiklanma event'i - ozel event gonderir
    badge.addEventListener('click', function () {
      var event = new CustomEvent('pazarradar:badge-click', { bubbles: true });
      parts.host.dispatchEvent(event);
    });

    parts.shadow.appendChild(badge);
    fadeIn(parts.host);
    return parts.host;
  }

  // ─── 2. Search Result Card ─────────────────────────────────

  /**
   * Arama sonuclari sayfasinda her urunun altina eklenen kompakt analiz karti.
   *
   * @param {Object} data - Kart verileri
   * @param {number} data.monthlySales - Tahmini aylik satis adedi
   * @param {number} data.monthlyRevenue - Tahmini aylik ciro (TL)
   * @param {number} data.totalReviews - Toplam yorum sayisi
   * @param {number} data.monthlyReviews - Aylik yorum sayisi
   * @param {number} data.favorites - Favori sayisi
   * @param {number} data.commissionRate - Komisyon orani (%)
   * @param {number} data.netMargin - Tahmini net marj (%)
   * @param {string} data.trend - Trend yonu ('up'|'down'|'stable')
   * @returns {HTMLElement} Kart host elemani
   */
  function createSearchResultCard(data) {
    if (!data || typeof data !== 'object') {
      return createSimpleElement('search-card-error');
    }

    var css = [
      '.pr-search-card {',
      '  padding: 8px 12px;',
      '  margin-top: 4px;',
      '  font-size: 11px;',
      '  cursor: pointer;',
      '  transition: background-color 150ms ease, max-height 200ms ease;',
      '  max-height: 56px;',
      '  overflow: hidden;',
      '}',
      '.pr-search-card:hover {',
      '  background: rgba(255,255,255,0.08);',
      '}',
      '.pr-search-card.pr-expanded {',
      '  max-height: 300px;',
      '}',
      '.pr-card-header {',
      '  display: flex;',
      '  align-items: center;',
      '  justify-content: space-between;',
      '  margin-bottom: 4px;',
      '}',
      '.pr-card-grid {',
      '  display: grid;',
      '  grid-template-columns: 1fr 1fr;',
      '  gap: 2px 16px;',
      '}',
      '.pr-card-item {',
      '  display: flex;',
      '  align-items: center;',
      '  gap: 4px;',
      '  white-space: nowrap;',
      '}',
      '.pr-card-label {',
      '  color: var(--pr-text-muted);',
      '}',
      '.pr-card-value {',
      '  font-weight: 600;',
      '  color: var(--pr-text);',
      '}',
      '.pr-card-trend {',
      '  display: inline-flex;',
      '  align-items: center;',
      '  gap: 3px;',
      '  font-weight: 600;',
      '}',
      '.pr-expand-hint {',
      '  text-align: center;',
      '  color: var(--pr-text-muted);',
      '  font-size: 10px;',
      '  margin-top: 4px;',
      '}',
      '.pr-expand-section {',
      '  margin-top: 6px;',
      '  padding-top: 6px;',
      '  border-top: 1px solid var(--pr-border);',
      '  font-size: 11px;',
      '  text-align: center;',
      '  color: var(--pr-text-muted);',
      '}'
    ].join('\n');

    var parts = createShadowHost('search-card', css);
    var card = el('div', { className: 'pr-container pr-glass pr-search-card' });

    var trendInfo = getTrendInfo(data.trend);
    var marginColor = getMarginColor(data.netMargin);

    // Kart baslik satiri
    var header = el('div', { className: 'pr-card-header' });
    var logo = createBadgeLogo();

    var trendSpan = el('span', {
      className: 'pr-card-trend',
      style: 'color:' + trendInfo.color
    }, trendInfo.icon + ' ' + trendInfo.label);

    header.appendChild(logo);
    header.appendChild(trendSpan);

    // Veri gridi
    var grid = el('div', { className: 'pr-card-grid' });

    var gridItems = [
      { label: 'Satis:', value: '~' + formatCompact(data.monthlySales) + '/ay' },
      { label: 'Ciro:', value: '~' + formatCompact(data.monthlyRevenue) + ' TL' },
      { label: 'Yorum:', value: formatNumber(data.totalReviews) + ' (' + formatNumber(data.monthlyReviews) + '/ay)' },
      { label: 'Favori:', value: formatCompact(data.favorites) },
      { label: 'Komisyon:', value: formatPercent(data.commissionRate) },
      { label: 'Net Marj:', value: formatPercent(data.netMargin), color: marginColor }
    ];

    for (var i = 0; i < gridItems.length; i++) {
      var item = el('div', { className: 'pr-card-item' });
      var labelEl = el('span', { className: 'pr-card-label' }, gridItems[i].label);
      var valueEl = el('span', {
        className: 'pr-card-value',
        style: gridItems[i].color ? 'color:' + gridItems[i].color : ''
      }, gridItems[i].value);
      item.appendChild(labelEl);
      item.appendChild(valueEl);
      grid.appendChild(item);
    }

    // Genisletilmis bolum (tiklaninca acilir)
    var expandSection = el('div', { className: 'pr-expand-section', style: 'display:none;' });
    var expandMsg = el('div', {
      style: 'color:' + TOKENS.textMuted + '; font-size:10px;'
    }, 'Detayli analiz icin urun sayfasini ziyaret edin');
    expandSection.appendChild(expandMsg);

    var expandHint = el('div', { className: 'pr-expand-hint' }, 'Detay icin tiklayin');

    card.appendChild(header);
    card.appendChild(grid);
    card.appendChild(expandSection);
    card.appendChild(expandHint);

    // Tiklanma: genislet/daralt
    card.addEventListener('click', function () {
      var isExpanded = card.classList.toggle('pr-expanded');
      expandSection.style.display = isExpanded ? 'block' : 'none';
      expandHint.textContent = isExpanded ? '' : 'Detay icin tiklayin';
    });

    parts.shadow.appendChild(card);
    fadeIn(parts.host);
    return parts.host;
  }

  // ─── 3. Search Aggregate Bar ───────────────────────────────

  /**
   * Arama sonuclari sayfasinin ustune yerlestirilen toplu analiz cubugu.
   * Sticky konum ile sayfa kaydirildikca gorunur kalir.
   *
   * @param {Object} data - Toplu veri
   * @param {string} data.category - Kategori adi
   * @param {number} data.avgMonthlySales - Ortalama aylik satis
   * @param {number} data.avgPrice - Ortalama fiyat (TL)
   * @param {number} data.totalSellers - Toplam satici sayisi
   * @param {number} data.nicheScore - Nis skoru (0-10)
   * @param {number} data.totalProducts - Toplam urun sayisi
   * @returns {HTMLElement} Cubuk host elemani
   */
  function createSearchAggregateBar(data) {
    if (!data || typeof data !== 'object') {
      return createSimpleElement('aggregate-bar-error');
    }

    var css = [
      '.pr-agg-wrap {',
      '  position: sticky;',
      '  top: 0;',
      '  left: 0;',
      '  right: 0;',
      '  z-index: ' + Z_INDEX + ';',
      '}',
      '.pr-agg-bar {',
      '  display: flex;',
      '  align-items: center;',
      '  justify-content: space-between;',
      '  padding: 10px 20px;',
      '  border-radius: 0 0 8px 8px;',
      '}',
      '.pr-agg-main {',
      '  flex: 1;',
      '  display: flex;',
      '  flex-direction: column;',
      '  gap: 6px;',
      '}',
      '.pr-agg-top {',
      '  display: flex;',
      '  align-items: center;',
      '  gap: 12px;',
      '}',
      '.pr-agg-stats {',
      '  display: flex;',
      '  align-items: center;',
      '  flex-wrap: wrap;',
      '  gap: 6px 20px;',
      '  overflow: hidden;',
      '  transition: max-height 200ms ease, opacity 200ms ease;',
      '  max-height: 100px;',
      '  opacity: 1;',
      '}',
      '.pr-agg-stats.pr-collapsed {',
      '  max-height: 0;',
      '  opacity: 0;',
      '}',
      '.pr-agg-stat {',
      '  display: flex;',
      '  align-items: center;',
      '  gap: 6px;',
      '  font-size: 12px;',
      '  white-space: nowrap;',
      '}',
      '.pr-agg-stat-label {',
      '  color: var(--pr-text-muted);',
      '}',
      '.pr-agg-stat-value {',
      '  font-weight: 700;',
      '  color: var(--pr-text);',
      '}',
      '.pr-agg-toggle {',
      '  background: none;',
      '  border: none;',
      '  color: var(--pr-text-muted);',
      '  cursor: pointer;',
      '  font-size: 16px;',
      '  padding: 2px 6px;',
      '  transition: color 150ms ease;',
      '}',
      '.pr-agg-toggle:hover { color: var(--pr-text); }'
    ].join('\n');

    var parts = createShadowHost('aggregate-bar', css);
    var container = el('div', { className: 'pr-container pr-agg-wrap' });
    var bar = el('div', { className: 'pr-glass pr-agg-bar' });

    // Sol taraf
    var mainContent = el('div', { className: 'pr-agg-main' });

    // Ust satir: logo + kategori
    var topRow = el('div', { className: 'pr-agg-top' });
    topRow.appendChild(createBadgeLogo());
    topRow.appendChild(buildStatDivider());

    var catStat = el('span', { className: 'pr-agg-stat' });
    catStat.appendChild(el('span', { className: 'pr-agg-stat-label' }, 'Kategori:'));
    catStat.appendChild(el('span', { className: 'pr-agg-stat-value' }, data.category || '-'));
    topRow.appendChild(catStat);

    mainContent.appendChild(topRow);

    // Stats satirlari
    var statsBody = el('div', { className: 'pr-agg-stats' });

    var scoreColor = getScoreColor(data.nicheScore);

    var statItems = [
      { label: 'Ort Satis:', value: formatNumber(data.avgMonthlySales) + '/ay', color: null },
      { label: 'Ort Fiyat:', value: formatCurrency(data.avgPrice, 0), color: null },
      { label: 'Satici:', value: formatNumber(data.totalSellers), color: null },
      { label: 'Urun:', value: formatNumber(data.totalProducts), color: null },
      { label: 'Nis:', value: (data.nicheScore != null ? data.nicheScore.toFixed(1) : '-') + '/5', color: scoreColor }
    ];

    for (var i = 0; i < statItems.length; i++) {
      if (i > 0) {
        statsBody.appendChild(buildStatDivider());
      }
      var stat = el('span', { className: 'pr-agg-stat' });
      stat.appendChild(el('span', { className: 'pr-agg-stat-label' }, statItems[i].label));
      stat.appendChild(el('span', {
        className: 'pr-agg-stat-value',
        style: statItems[i].color ? 'color:' + statItems[i].color : ''
      }, statItems[i].value));
      statsBody.appendChild(stat);
    }

    mainContent.appendChild(statsBody);
    bar.appendChild(mainContent);

    // Collapse/expand butonu
    var toggleBtn = el('button', { className: 'pr-agg-toggle', title: 'Daralt' }, '\u25B2');
    var isCollapsed = false;
    toggleBtn.addEventListener('click', function () {
      isCollapsed = !isCollapsed;
      statsBody.classList.toggle('pr-collapsed', isCollapsed);
      toggleBtn.textContent = isCollapsed ? '\u25BC' : '\u25B2';
      toggleBtn.title = isCollapsed ? 'Genislet' : 'Daralt';
    });

    bar.appendChild(toggleBtn);
    container.appendChild(bar);
    parts.shadow.appendChild(container);
    fadeIn(parts.host);
    return parts.host;
  }

  // ─── 4. Product Detail Panel ───────────────────────────────

  /**
   * Urun detay sayfasinda sag tarafta gorunen kapsamli analiz paneli.
   * Surukleme, kapatma, ve bolum acma/kapama ozellikleri icerir.
   *
   * @param {Object} data - Panel verileri
   * @param {number} data.productScore - Urun skoru (0-5)
   * @param {number} data.estimatedMonthlySales - Tahmini aylik satis
   * @param {number} data.estimatedMonthlyRevenue - Tahmini aylik ciro
   * @param {number} data.reviewRate - Aylik yorum hizi
   * @param {number} data.favoriteRate - Aylik favori hizi
   * @param {string} data.trend - Trend yonu ('up'|'down'|'stable')
   * @param {number} data.salePrice - Satis fiyati (TL)
   * @param {number} data.commissionRate - Komisyon orani (%)
   * @param {number} data.commissionAmount - Komisyon tutari (TL)
   * @param {number} data.commissionKDV - Komisyon KDV (TL)
   * @param {number} data.platformFee - Platform ucreti (TL)
   * @param {number} data.cargoFee - Kargo ucreti (TL)
   * @param {number} data.netRevenue - Net gelir (TL)
   * @param {Object} [data.comparison] - Platform karsilastirma verisi (compareAll sonucu)
   * @param {Array} [data.priceHistory] - Fiyat gecmisi (createPriceChart verisi)
   * @param {boolean} [data.isPro=false] - Kullanici PRO mu
   * @param {boolean} [data.limitReached=false] - Gunluk limit doldu mu
   * @param {Object} [data.limitInfo] - Limit bilgisi {current, limit, remaining}
   * @returns {HTMLElement} Panel host elemani
   */

  /**
   * Limit doldu karti olusturur. Inline lisans girisi icindedir.
   * @param {Object} data - Panel verisi (limitInfo icermeli)
   * @param {HTMLElement} hostEl - Shadow host (reload icin)
   * @returns {HTMLElement}
   */
  function buildLimitReachedCard(data, hostEl) {
    var info = data.limitInfo || {};
    var current = info.current || info.limit || 5;
    var limit = info.limit || 5;

    var card = el('div', {
      className: 'pr-limit-card',
      style: [
        'background: linear-gradient(135deg, rgba(239,68,68,0.12), rgba(239,68,68,0.04))',
        'border: 1px solid rgba(239,68,68,0.3)',
        'border-radius: 12px',
        'padding: 16px',
        'text-align: center'
      ].join(';')
    });

    // Kilit ikonu + baslik
    var title = el('div', {
      style: 'font-size:16px;font-weight:700;color:#ef4444;margin-bottom:4px;'
    }, '\uD83D\uDD12 G\u00FCnl\u00FCk Limit Doldu (' + current + '/' + limit + ')');
    card.appendChild(title);

    // Aciklama
    var desc = el('div', {
      style: 'font-size:12px;color:' + TOKENS.textSecondary + ';margin-bottom:14px;'
    }, 'PRO ile s\u0131n\u0131rs\u0131z analiz yap\u0131n');
    card.appendChild(desc);

    // Lisans input alani
    var inputWrap = el('div', {
      style: 'display:flex;gap:6px;margin-bottom:10px;'
    });

    var licenseInput = el('input', {
      type: 'text',
      placeholder: 'PR-XXXX-XXXX-XXXX-XXXX',
      style: [
        'flex:1',
        'padding:8px 10px',
        'border:1px solid ' + TOKENS.border,
        'border-radius:8px',
        'background:rgba(255,255,255,0.06)',
        'color:' + TOKENS.text,
        'font-size:12px',
        'font-family:monospace',
        'outline:none'
      ].join(';'),
      ariaLabel: 'Lisans anahtari'
    });
    inputWrap.appendChild(licenseInput);
    card.appendChild(inputWrap);

    // Aktive Et butonu
    var activateBtn = el('button', {
      style: [
        'width:100%',
        'padding:10px',
        'border:none',
        'border-radius:8px',
        'background:linear-gradient(135deg, ' + TOKENS.accent + ', ' + TOKENS.accentLight + ')',
        'color:#fff',
        'font-weight:700',
        'font-size:13px',
        'cursor:pointer',
        'transition:opacity 0.2s'
      ].join(';')
    }, 'Aktive Et');

    // Sonuc mesaji alani
    var resultMsg = el('div', {
      style: 'font-size:11px;margin-top:8px;min-height:16px;'
    });

    activateBtn.addEventListener('click', function () {
      var key = licenseInput.value.trim();
      if (!key) {
        resultMsg.textContent = 'Lisans anahtarini girin';
        resultMsg.style.color = TOKENS.danger;
        return;
      }

      activateBtn.textContent = 'Kontrol ediliyor...';
      activateBtn.style.opacity = '0.6';
      activateBtn.disabled = true;

      try {
        chrome.runtime.sendMessage({
          type: 'activate_license',
          payload: { key: key }
        }, function (response) {
          if (chrome.runtime.lastError) {
            resultMsg.textContent = 'Baglanti hatasi';
            resultMsg.style.color = TOKENS.danger;
            activateBtn.textContent = 'Aktive Et';
            activateBtn.style.opacity = '1';
            activateBtn.disabled = false;
            return;
          }

          if (response && response.success) {
            resultMsg.textContent = '\u2705 PRO aktif! Sayfa yenileniyor...';
            resultMsg.style.color = TOKENS.success;
            setTimeout(function () {
              window.location.reload();
            }, 1200);
          } else {
            resultMsg.textContent = (response && response.error) || 'Gecersiz lisans anahtari';
            resultMsg.style.color = TOKENS.danger;
            activateBtn.textContent = 'Aktive Et';
            activateBtn.style.opacity = '1';
            activateBtn.disabled = false;
          }
        });
      } catch (_err) {
        resultMsg.textContent = 'Hata olustu';
        resultMsg.style.color = TOKENS.danger;
        activateBtn.textContent = 'Aktive Et';
        activateBtn.style.opacity = '1';
        activateBtn.disabled = false;
      }
    });

    card.appendChild(activateBtn);
    card.appendChild(resultMsg);

    // Link
    var link = el('div', {
      style: 'font-size:11px;color:' + TOKENS.textMuted + ';margin-top:10px;'
    }, 'veya: pazarradar.teknoweb.net');
    card.appendChild(link);

    return card;
  }

  function createProductPanel(data) {
    if (!data || typeof data !== 'object') {
      return createSimpleElement('panel-error');
    }

    var css = [
      '.pr-panel {',
      '  position: fixed;',
      '  top: 80px;',
      '  right: 16px;',
      '  width: ' + PANEL_WIDTH + 'px;',
      '  max-height: calc(100vh - 100px);',
      '  overflow-y: auto;',
      '  z-index: ' + Z_INDEX + ';',
      '  display: flex;',
      '  flex-direction: column;',
      '}',
      '.pr-panel::-webkit-scrollbar { width: 4px; }',
      '.pr-panel::-webkit-scrollbar-track { background: transparent; }',
      '.pr-panel::-webkit-scrollbar-thumb {',
      '  background: var(--pr-border);',
      '  border-radius: 2px;',
      '}',
      '.pr-panel-header {',
      '  display: flex;',
      '  align-items: center;',
      '  justify-content: space-between;',
      '  padding: 12px 14px;',
      '  border-bottom: 1px solid var(--pr-border);',
      '  flex-shrink: 0;',
      '}',
      '.pr-panel-body {',
      '  padding: 12px 14px;',
      '  display: flex;',
      '  flex-direction: column;',
      '  gap: 10px;',
      '}',
      '.pr-score-bar {',
      '  display: flex;',
      '  align-items: center;',
      '  gap: 10px;',
      '}',
      '.pr-score-value {',
      '  font-size: 28px;',
      '  font-weight: 800;',
      '  line-height: 1;',
      '}',
      '.pr-score-max {',
      '  font-size: 14px;',
      '  font-weight: 400;',
      '  color: var(--pr-text-muted);',
      '}',
      '.pr-score-track {',
      '  flex: 1;',
      '  height: 8px;',
      '  background: rgba(255,255,255,0.1);',
      '  border-radius: 4px;',
      '  overflow: hidden;',
      '}',
      '.pr-score-fill {',
      '  height: 100%;',
      '  border-radius: 4px;',
      '  transition: width 400ms ease;',
      '}',
      '.pr-section { padding: 0; }',
      '.pr-section-header {',
      '  cursor: pointer;',
      '  user-select: none;',
      '  display: flex;',
      '  align-items: center;',
      '  justify-content: space-between;',
      '  padding: 4px 0;',
      '}',
      '.pr-section-chevron {',
      '  transition: transform 200ms ease;',
      '  font-size: 10px;',
      '  color: var(--pr-text-muted);',
      '}',
      '.pr-section-header.pr-collapsed .pr-section-chevron {',
      '  transform: rotate(-90deg);',
      '}',
      '.pr-section-content {',
      '  overflow: hidden;',
      '  transition: max-height 200ms ease;',
      '}',
      '.pr-data-row {',
      '  display: flex;',
      '  justify-content: space-between;',
      '  align-items: center;',
      '  padding: 3px 0;',
      '  font-size: 12px;',
      '}',
      '.pr-data-label {',
      '  color: var(--pr-text-secondary);',
      '}',
      '.pr-data-value {',
      '  font-weight: 600;',
      '  text-align: right;',
      '}',
      '.pr-data-separator {',
      '  border: none;',
      '  border-top: 1px dashed var(--pr-border);',
      '  margin: 4px 0;',
      '}',
      '.pr-data-total {',
      '  font-size: 14px;',
      '  font-weight: 700;',
      '}',
      '.pr-pro-lock {',
      '  display: flex;',
      '  align-items: center;',
      '  justify-content: center;',
      '  gap: 6px;',
      '  padding: 10px;',
      '  border-radius: 6px;',
      '  background: linear-gradient(135deg, rgba(249,115,22,0.15), rgba(249,115,22,0.05));',
      '  border: 1px solid rgba(249,115,22,0.25);',
      '  color: var(--pr-accent);',
      '  font-size: 12px;',
      '  font-weight: 600;',
      '  cursor: pointer;',
      '  transition: background 150ms ease;',
      '}',
      '.pr-pro-lock:hover {',
      '  background: linear-gradient(135deg, rgba(249,115,22,0.25), rgba(249,115,22,0.1));',
      '}',
      '.pr-cost-input-row {',
      '  display: flex;',
      '  align-items: center;',
      '  gap: 8px;',
      '  margin-top: 4px;',
      '}',
      '.pr-cost-input-row .pr-input {',
      '  width: 120px;',
      '  flex-shrink: 0;',
      '}',
      '.pr-cost-result {',
      '  margin-top: 6px;',
      '  padding: 8px;',
      '  border-radius: 6px;',
      '}',
      '.pr-chart-container {',
      '  width: 100%;',
      '  height: 120px;',
      '  position: relative;',
      '}',
      /* Karsilastirma tablosu - panel icinde de kullanilir */
      '.pr-comp-table {',
      '  width: 100%;',
      '  border-collapse: collapse;',
      '}',
      '.pr-comp-table th, .pr-comp-table td {',
      '  text-align: center;',
      '  padding: 6px 8px;',
      '  font-size: 12px;',
      '  border-bottom: 1px solid var(--pr-border);',
      '}',
      '.pr-comp-table th {',
      '  font-weight: 700;',
      '  color: var(--pr-text-secondary);',
      '  font-size: 11px;',
      '  text-transform: uppercase;',
      '  letter-spacing: 0.3px;',
      '}',
      '.pr-comp-best {',
      '  border: 1px solid ' + TOKENS.success + ' !important;',
      '  border-radius: 6px;',
      '  background: rgba(34, 197, 94, 0.08);',
      '}',
      '.pr-comp-star { color: ' + TOKENS.accentLight + '; margin-left: 2px; }',
      '.pr-comp-revenue { font-weight: 700; font-size: 14px; }',
      '.pr-comp-margin { font-weight: 600; }',
      /* Blur/tease - FREE kullanicilar icin */
      '.pr-blur-value {',
      '  filter: blur(6px);',
      '  user-select: none;',
      '  pointer-events: none;',
      '}',
      '.pr-trial-banner {',
      '  background: linear-gradient(135deg, rgba(34,197,94,0.15), rgba(34,197,94,0.05));',
      '  border: 1px solid rgba(34,197,94,0.3);',
      '  border-radius: 8px;',
      '  padding: 8px 12px;',
      '  font-size: 11px;',
      '  color: #22c55e;',
      '  text-align: center;',
      '  margin-bottom: 8px;',
      '}',
      '.pr-locked-msg {',
      '  display: flex;',
      '  align-items: center;',
      '  justify-content: center;',
      '  gap: 6px;',
      '  padding: 12px;',
      '  border-radius: 6px;',
      '  background: linear-gradient(135deg, rgba(249,115,22,0.1), rgba(249,115,22,0.03));',
      '  border: 1px dashed rgba(249,115,22,0.3);',
      '  color: var(--pr-accent);',
      '  font-size: 11px;',
      '  font-weight: 600;',
      '  cursor: pointer;',
      '}',
      '.pr-locked-msg:hover {',
      '  background: linear-gradient(135deg, rgba(249,115,22,0.18), rgba(249,115,22,0.06));',
      '}'
    ].join('\n');

    var parts = createShadowHost('product-panel', css);
    var panel = el('div', { className: 'pr-container pr-glass pr-panel' });

    // ─── Panel Header ─────────
    var header = el('div', { className: 'pr-panel-header' });
    header.appendChild(createBadgeLogoWithText('PazarRadar Analiz'));
    header.appendChild(buildCloseButton(function () {
      fadeOut(parts.host);
    }));

    // ─── Panel Body ───────────
    var body = el('div', { className: 'pr-panel-body' });

    // --- URUN SKORU ---
    body.appendChild(buildScoreSection(data.productScore));

    // --- TRIAL BANNER ---
    if (data.trialDaysLeft != null && data.trialDaysLeft > 0) {
      body.appendChild(el('div', { className: 'pr-trial-banner' },
        '\u2728 PRO Deneme: ' + data.trialDaysLeft + ' g\u00fcn kald\u0131'));
    }

    body.appendChild(createDivider());

    // --- LIMIT KONTROLU ---
    if (data.limitReached) {
      // Limit dolmus - kisitli icerik goster
      body.appendChild(createDivider());
      body.appendChild(buildLimitReachedCard(data, parts.host));
    } else {
      // --- TALEP ANALIZI ---
      var demandSection = buildCollapsibleSection('TALEP ANAL\u0130Z\u0130', function (content) {
        var trendInfo = getTrendInfo(data.trend);

        var rows = [
          { label: 'Tahmini Satis', value: '~' + formatNumber(data.estimatedMonthlySales) + '/ay', blur: true },
          { label: 'Tahmini Ciro', value: formatCompact(data.estimatedMonthlyRevenue) + ' TL/ay', blur: true },
          { label: 'Yorum Hizi', value: formatNumber(data.reviewRate) + '/ay', blur: true },
          { label: 'Favori Hizi', value: formatNumber(data.favoriteRate) + '/ay', blur: true },
          { label: 'Trend', value: trendInfo.icon + ' ' + trendInfo.label, color: trendInfo.color, blur: false }
        ];

        for (var i = 0; i < rows.length; i++) {
          if (rows[i].blur) {
            content.appendChild(buildBlurredDataRow(rows[i].label, rows[i].value, data.isPro, rows[i].color));
          } else {
            content.appendChild(buildDataRow(rows[i].label, rows[i].value, rows[i].color));
          }
        }
      });
      body.appendChild(demandSection);
      body.appendChild(createDivider());

      // --- KAR HESAPLAMA ---
      var profitSection = buildCollapsibleSection('KAR HESAPLAMA', function (content) {
        var feeRows = [
          { label: 'Satis Fiyati', value: formatCurrency(data.salePrice) },
          { label: 'Komisyon (' + formatPercent(data.commissionRate) + ')', value: '-' + formatCurrency(data.commissionAmount) },
          { label: 'Komisyon KDV', value: '-' + formatCurrency(data.commissionKDV) },
          { label: 'Platform Bedeli', value: '-' + formatCurrency(data.platformFee) },
          { label: 'Kargo', value: '-' + formatCurrency(data.cargoFee) }
        ];

        for (var i = 0; i < feeRows.length; i++) {
          content.appendChild(buildDataRow(feeRows[i].label, feeRows[i].value));
        }

        content.appendChild(buildSeparator());

        var netRow = buildDataRow('Net Gelir', formatCurrency(data.netRevenue));
        var netValueEl = netRow.querySelector('.pr-data-value');
        if (netValueEl) {
          netValueEl.classList.add('pr-data-total');
          netValueEl.style.color = TOKENS.accent;
        }
        content.appendChild(netRow);

        // Maliyet girisi alani
        var costRow = el('div', { className: 'pr-cost-input-row' });
        costRow.appendChild(el('span', { className: 'pr-data-label' }, 'Maliyetiniz:'));

        var costInput = el('input', {
          className: 'pr-input',
          type: 'number',
          placeholder: '0,00',
          min: '0',
          step: '0.01',
          ariaLabel: 'Urun maliyeti (TL)'
        });
        costRow.appendChild(costInput);
        costRow.appendChild(el('span', { className: 'pr-data-label' }, 'TL'));
        content.appendChild(costRow);

        // Dinamik kar sonuc alani
        var costResult = el('div', {
          className: 'pr-cost-result pr-glass-light',
          style: 'display:none;'
        });
        content.appendChild(costResult);

        // Maliyet girilince anlik hesaplama
        costInput.addEventListener('input', function () {
          var costVal = parseFloat(costInput.value);
          if (!Number.isFinite(costVal) || costVal < 0) {
            costResult.style.display = 'none';
            return;
          }

          var netProfit = (data.netRevenue || 0) - costVal;
          var marginPercent = (data.salePrice > 0) ? ((netProfit / data.salePrice) * 100) : 0;
          var profitColor = netProfit >= 0 ? TOKENS.success : TOKENS.danger;

          costResult.style.display = 'block';
          // Guvenli DOM temizleme ve yeniden olusturma
          while (costResult.firstChild) {
            costResult.removeChild(costResult.firstChild);
          }

          var profitRow = buildDataRow('NET KAR', formatCurrency(netProfit), profitColor);
          var profitValueEl = profitRow.querySelector('.pr-data-value');
          if (profitValueEl) {
            profitValueEl.classList.add('pr-data-total');
          }
          costResult.appendChild(profitRow);
          costResult.appendChild(buildDataRow('Kar Marji', formatPercent(marginPercent), profitColor));
        });
      });
      body.appendChild(profitSection);
      body.appendChild(createDivider());

      // --- PLATFORM KARSILASTIRMA ---
      if (data.comparison) {
        var compSection = buildCollapsibleSection('PLATFORM KAR\u015EILA\u015ETIRMA', function (content) {
          content.appendChild(buildComparisonTable(data.comparison, data.isPro));
        });
        body.appendChild(compSection);
        body.appendChild(createDivider());
      }

      // --- FIYAT GECMISI ---
      if (data.priceHistory && data.priceHistory.length > 0) {
        if (data.isPro) {
          // PRO/Trial: tam grafik goster
          var chartSection = buildCollapsibleSection('F\u0130YAT GE\u00C7M\u0130\u015E\u0130', function (content) {
            var chartContainer = el('div', { className: 'pr-chart-container' });
            content.appendChild(chartContainer);

            requestAnimationFrame(function () {
              var chartCanvas = buildPriceChartCanvas(data.priceHistory, chartContainer);
              if (chartCanvas) {
                chartContainer.appendChild(chartCanvas);
              }
            });
          });
          body.appendChild(chartSection);
        } else {
          // FREE: kilitli fiyat gecmisi
          var lockedChart = buildCollapsibleSection('F\u0130YAT GE\u00C7M\u0130\u015E\u0130', function (content) {
            var lockMsg = el('div', { className: 'pr-locked-msg' },
              '\uD83D\uDD12 Fiyat ge\u00e7mi\u015fini g\u00f6rmek i\u00e7in PRO\'ya y\u00fckseltin');
            lockMsg.addEventListener('click', function () {
              var event = new CustomEvent('pazarradar:upgrade-click', { bubbles: true });
              parts.host.dispatchEvent(event);
            });
            content.appendChild(lockMsg);
          });
          body.appendChild(lockedChart);
        }
        body.appendChild(createDivider());
      }

      // --- PRO KILIDI --- (trial kullanicilara gosterme)
      if (!data.isPro && !(data.trialDaysLeft > 0)) {
        var proLock = el('div', { className: 'pr-pro-lock' }, '\uD83D\uDD12 PRO: Tam Analiz');
        proLock.addEventListener('click', function () {
          var event = new CustomEvent('pazarradar:upgrade-click', { bubbles: true });
          parts.host.dispatchEvent(event);
        });
        body.appendChild(proLock);
      }
    }

    panel.appendChild(header);
    panel.appendChild(body);

    // Surukleme ozelligi
    makeDraggable(header, panel);

    parts.shadow.appendChild(panel);
    fadeIn(parts.host);
    return parts.host;
  }

  // ─── 5. Profit Mini Calculator ─────────────────────────────

  /**
   * Kompakt satirici kar hesaplayici widget'i.
   * Platform, komisyon ve masraflar hesaplanarak anlik guncellenir.
   *
   * @param {Object} data - Hesaplayici verisi
   * @param {string} data.platform - Platform adi ('trendyol'|'hepsiburada'|'n11')
   * @param {number} data.salePrice - Satis fiyati (TL)
   * @param {number} data.commissionRate - Komisyon orani (%)
   * @param {number} data.commissionAmount - Komisyon tutari (TL)
   * @param {number} data.commissionKDV - Komisyon KDV (TL)
   * @param {number} data.platformFee - Platform bedeli (TL)
   * @param {number} data.platformFeeKDV - Platform bedeli KDV (TL)
   * @param {number} data.cargoFee - Kargo ucreti (TL)
   * @param {number} data.netRevenue - Net gelir (TL)
   * @returns {HTMLElement} Hesaplayici host elemani
   */
  function createProfitMiniCalc(data) {
    if (!data || typeof data !== 'object') {
      return createSimpleElement('mini-calc-error');
    }

    var css = [
      '.pr-mini-calc {',
      '  padding: 12px;',
      '  width: 280px;',
      '  font-size: 12px;',
      '}',
      '.pr-calc-header {',
      '  display: flex;',
      '  align-items: center;',
      '  justify-content: space-between;',
      '  margin-bottom: 8px;',
      '}',
      '.pr-calc-platform {',
      '  background: var(--pr-bg-glass);',
      '  border: 1px solid var(--pr-border);',
      '  border-radius: 4px;',
      '  padding: 2px 8px;',
      '  font-size: 11px;',
      '  color: var(--pr-accent);',
      '  font-weight: 600;',
      '}',
      '.pr-calc-row {',
      '  display: flex;',
      '  justify-content: space-between;',
      '  align-items: center;',
      '  padding: 2px 0;',
      '}',
      '.pr-calc-label { color: var(--pr-text-secondary); }',
      '.pr-calc-value {',
      '  font-weight: 600;',
      '  font-variant-numeric: tabular-nums;',
      '}',
      '.pr-calc-fee { color: var(--pr-danger); }',
      '.pr-calc-total {',
      '  font-size: 14px;',
      '  font-weight: 700;',
      '}',
      '.pr-calc-separator {',
      '  border: none;',
      '  border-top: 1px dashed var(--pr-border);',
      '  margin: 6px 0;',
      '}',
      '.pr-calc-cost-row {',
      '  display: flex;',
      '  align-items: center;',
      '  gap: 8px;',
      '  margin-top: 8px;',
      '}',
      '.pr-calc-cost-row .pr-input { width: 100px; }',
      '.pr-calc-result {',
      '  margin-top: 6px;',
      '  padding: 8px;',
      '  border-radius: 6px;',
      '  text-align: center;',
      '}'
    ].join('\n');

    var parts = createShadowHost('mini-calc', css);
    var calc = el('div', { className: 'pr-container pr-glass pr-mini-calc' });

    // Platform baslik satiri
    var platformNames = {
      trendyol: 'Trendyol',
      hepsiburada: 'Hepsiburada',
      n11: 'N11'
    };

    var calcHeader = el('div', { className: 'pr-calc-header' });
    calcHeader.appendChild(createBadgeLogoWithText('Kar Hesap'));
    calcHeader.appendChild(el('span', { className: 'pr-calc-platform' }, platformNames[data.platform] || data.platform));
    calc.appendChild(calcHeader);

    // Kesinti satirlari
    var breakdownItems = [
      { label: 'Satis Fiyati', value: formatCurrency(data.salePrice), isFee: false },
      { label: 'Komisyon (' + formatPercent(data.commissionRate) + ')', value: '-' + formatCurrency(data.commissionAmount), isFee: true },
      { label: 'Komisyon KDV', value: '-' + formatCurrency(data.commissionKDV), isFee: true },
      { label: 'Platform Bedeli', value: '-' + formatCurrency(data.platformFee), isFee: true },
      { label: 'Platform KDV', value: '-' + formatCurrency(data.platformFeeKDV), isFee: true },
      { label: 'Kargo', value: '-' + formatCurrency(data.cargoFee), isFee: true }
    ];

    for (var i = 0; i < breakdownItems.length; i++) {
      var row = el('div', { className: 'pr-calc-row' });
      row.appendChild(el('span', { className: 'pr-calc-label' }, breakdownItems[i].label));
      row.appendChild(el('span', {
        className: 'pr-calc-value' + (breakdownItems[i].isFee ? ' pr-calc-fee' : '')
      }, breakdownItems[i].value));
      calc.appendChild(row);
    }

    // Separator + Net gelir
    calc.appendChild(el('hr', { className: 'pr-calc-separator' }));

    var netRow = el('div', { className: 'pr-calc-row' });
    netRow.appendChild(el('span', { className: 'pr-calc-label pr-calc-total' }, 'Net Gelir'));
    netRow.appendChild(el('span', {
      className: 'pr-calc-value pr-calc-total',
      style: 'color:' + TOKENS.accent
    }, formatCurrency(data.netRevenue)));
    calc.appendChild(netRow);

    // Maliyet girisi
    var costRow = el('div', { className: 'pr-calc-cost-row' });
    costRow.appendChild(el('span', { className: 'pr-calc-label' }, 'Maliyetiniz:'));

    var costInput = el('input', {
      className: 'pr-input',
      type: 'number',
      placeholder: '0',
      min: '0',
      step: '0.01',
      ariaLabel: 'Urun maliyeti (TL)'
    });
    costRow.appendChild(costInput);
    costRow.appendChild(el('span', { className: 'pr-calc-label' }, 'TL'));
    calc.appendChild(costRow);

    // Sonuc alani
    var resultDiv = el('div', {
      className: 'pr-calc-result pr-glass-light',
      style: 'display:none;'
    });
    calc.appendChild(resultDiv);

    // Anlik hesaplama
    costInput.addEventListener('input', function () {
      var costVal = parseFloat(costInput.value);
      if (!Number.isFinite(costVal) || costVal < 0) {
        resultDiv.style.display = 'none';
        return;
      }

      var netProfit = (data.netRevenue || 0) - costVal;
      var marginPercent = (data.salePrice > 0) ? ((netProfit / data.salePrice) * 100) : 0;
      var profitColor = netProfit >= 0 ? TOKENS.success : TOKENS.danger;

      resultDiv.style.display = 'block';
      // Guvenli DOM temizleme
      while (resultDiv.firstChild) {
        resultDiv.removeChild(resultDiv.firstChild);
      }

      var profitResultRow = el('div', { className: 'pr-calc-row' });
      profitResultRow.appendChild(el('span', { className: 'pr-calc-label pr-calc-total' }, 'NET KAR'));
      profitResultRow.appendChild(el('span', {
        className: 'pr-calc-value pr-calc-total',
        style: 'color:' + profitColor
      }, formatCurrency(netProfit)));
      resultDiv.appendChild(profitResultRow);

      var marginResultRow = el('div', { className: 'pr-calc-row' });
      marginResultRow.appendChild(el('span', { className: 'pr-calc-label' }, 'Kar Marji'));
      marginResultRow.appendChild(el('span', {
        className: 'pr-calc-value',
        style: 'color:' + profitColor
      }, formatPercent(marginPercent)));
      resultDiv.appendChild(marginResultRow);
    });

    parts.shadow.appendChild(calc);
    fadeIn(parts.host);
    return parts.host;
  }

  // ─── 6. Price History Chart ────────────────────────────────

  /**
   * Fiyat gecmisini gosteren Canvas bazli mini grafik widget'i.
   * Harici kutuphane gerektirmez, tamamen Canvas 2D API ile cizilir.
   *
   * @param {Array<Object>} priceHistory - Fiyat verileri
   * @param {string|Date} priceHistory[].date - Tarih
   * @param {number} priceHistory[].price - Fiyat (TL)
   * @returns {HTMLElement} Grafik host elemani
   */
  function createPriceChart(priceHistory) {
    if (!Array.isArray(priceHistory) || priceHistory.length === 0) {
      return createSimpleElement('price-chart-error');
    }

    var css = [
      '.pr-chart-widget {',
      '  padding: 12px;',
      '  width: 100%;',
      '}',
      '.pr-chart-header {',
      '  display: flex;',
      '  justify-content: space-between;',
      '  align-items: center;',
      '  flex-wrap: wrap;',
      '  gap: 8px;',
      '  margin-bottom: 8px;',
      '}',
      '.pr-chart-stats {',
      '  display: flex;',
      '  gap: 12px;',
      '  font-size: 11px;',
      '}',
      '.pr-chart-stat {',
      '  display: flex;',
      '  gap: 4px;',
      '}',
      '.pr-chart-stat-label { color: var(--pr-text-muted); }',
      '.pr-chart-stat-value { font-weight: 600; }',
      '.pr-chart-canvas-wrap {',
      '  width: 100%;',
      '  height: 120px;',
      '  position: relative;',
      '}',
      'canvas {',
      '  display: block;',
      '  width: 100%;',
      '  height: 100%;',
      '}'
    ].join('\n');

    var parts = createShadowHost('price-chart', css);
    var widget = el('div', { className: 'pr-container pr-glass pr-chart-widget' });

    // Istatistik degerleri hesapla
    var prices = priceHistory.map(function (p) { return p.price; });
    var minPrice = Math.min.apply(null, prices);
    var maxPrice = Math.max.apply(null, prices);
    var avgPrice = prices.reduce(function (s, p) { return s + p; }, 0) / prices.length;

    // Baslik
    var headerDiv = el('div', { className: 'pr-chart-header' });
    headerDiv.appendChild(createBadgeLogoWithText('Fiyat Gecmisi'));

    var statsDiv = el('div', { className: 'pr-chart-stats' });

    // Min stat
    var minStat = el('span', { className: 'pr-chart-stat' });
    minStat.appendChild(el('span', { className: 'pr-chart-stat-label' }, 'Min:'));
    minStat.appendChild(el('span', { className: 'pr-chart-stat-value', style: 'color:' + TOKENS.success }, formatCurrency(minPrice, 0)));
    statsDiv.appendChild(minStat);

    // Ort stat
    var avgStat = el('span', { className: 'pr-chart-stat' });
    avgStat.appendChild(el('span', { className: 'pr-chart-stat-label' }, 'Ort:'));
    avgStat.appendChild(el('span', { className: 'pr-chart-stat-value' }, formatCurrency(avgPrice, 0)));
    statsDiv.appendChild(avgStat);

    // Max stat
    var maxStat = el('span', { className: 'pr-chart-stat' });
    maxStat.appendChild(el('span', { className: 'pr-chart-stat-label' }, 'Max:'));
    maxStat.appendChild(el('span', { className: 'pr-chart-stat-value', style: 'color:' + TOKENS.danger }, formatCurrency(maxPrice, 0)));
    statsDiv.appendChild(maxStat);

    headerDiv.appendChild(statsDiv);
    widget.appendChild(headerDiv);

    // Canvas sarmalayici
    var canvasWrap = el('div', { className: 'pr-chart-canvas-wrap' });
    widget.appendChild(canvasWrap);

    parts.shadow.appendChild(widget);

    // Canvas'i bir sonraki frame'de olustur
    requestAnimationFrame(function () {
      var canvas = buildPriceChartCanvas(priceHistory, canvasWrap);
      if (canvas) {
        canvasWrap.appendChild(canvas);
      }
    });

    fadeIn(parts.host);
    return parts.host;
  }

  /**
   * Canvas 2D API ile fiyat gecmisi cizgisini render eder.
   * @param {Array<Object>} priceHistory - Fiyat verileri
   * @param {HTMLElement} container - Canvas'in yerlestirilecegi konteyner
   * @returns {HTMLCanvasElement|null}
   */
  function buildPriceChartCanvas(priceHistory, container) {
    if (!priceHistory || priceHistory.length < 2 || !container) {
      return null;
    }

    var rect = container.getBoundingClientRect();
    var dpr = window.devicePixelRatio || 1;
    var cWidth = Math.max(rect.width || 260, 100);
    var cHeight = Math.max(rect.height || 120, 60);

    var canvas = document.createElement('canvas');
    canvas.width = cWidth * dpr;
    canvas.height = cHeight * dpr;
    canvas.style.width = cWidth + 'px';
    canvas.style.height = cHeight + 'px';

    var ctx = canvas.getContext('2d');
    if (!ctx) {
      return null;
    }

    ctx.scale(dpr, dpr);

    // Cizerken kullanilan padding
    var padTop = 10;
    var padBottom = 22;
    var padLeft = 8;
    var padRight = 8;

    var chartW = cWidth - padLeft - padRight;
    var chartH = cHeight - padTop - padBottom;

    var prices = priceHistory.map(function (p) { return p.price; });
    var minP = Math.min.apply(null, prices);
    var maxP = Math.max.apply(null, prices);
    var priceRange = maxP - minP || 1;
    var dataLen = priceHistory.length;

    /**
     * Veri indeksini X koordinatina cevirir.
     * @param {number} idx
     * @returns {number}
     */
    function xOf(idx) {
      return padLeft + (idx / (dataLen - 1)) * chartW;
    }

    /**
     * Fiyat degerini Y koordinatina cevirir.
     * @param {number} price
     * @returns {number}
     */
    function yOf(price) {
      return padTop + chartH - ((price - minP) / priceRange) * chartH;
    }

    // Arka plan kizgara cizgileri (3 yatay)
    ctx.strokeStyle = 'rgba(255,255,255,0.06)';
    ctx.lineWidth = 1;
    for (var g = 0; g < 3; g++) {
      var gy = padTop + (chartH / 2) * g;
      ctx.beginPath();
      ctx.moveTo(padLeft, gy);
      ctx.lineTo(padLeft + chartW, gy);
      ctx.stroke();
    }

    // Gradient dolgu alani
    var gradient = ctx.createLinearGradient(0, padTop, 0, padTop + chartH);
    gradient.addColorStop(0, 'rgba(249, 115, 22, 0.25)');
    gradient.addColorStop(1, 'rgba(249, 115, 22, 0.02)');

    // Alan (fill)
    ctx.beginPath();
    ctx.moveTo(xOf(0), yOf(prices[0]));
    for (var fi = 1; fi < dataLen; fi++) {
      ctx.lineTo(xOf(fi), yOf(prices[fi]));
    }
    ctx.lineTo(xOf(dataLen - 1), padTop + chartH);
    ctx.lineTo(xOf(0), padTop + chartH);
    ctx.closePath();
    ctx.fillStyle = gradient;
    ctx.fill();

    // Cizgi
    ctx.beginPath();
    ctx.moveTo(xOf(0), yOf(prices[0]));
    for (var li = 1; li < dataLen; li++) {
      ctx.lineTo(xOf(li), yOf(prices[li]));
    }
    ctx.strokeStyle = TOKENS.accent;
    ctx.lineWidth = 2;
    ctx.lineJoin = 'round';
    ctx.lineCap = 'round';
    ctx.stroke();

    // Son noktayi vurgula (mevcut fiyat)
    var lastX = xOf(dataLen - 1);
    var lastY = yOf(prices[dataLen - 1]);

    // Dis halka
    ctx.beginPath();
    ctx.arc(lastX, lastY, 5, 0, Math.PI * 2);
    ctx.fillStyle = 'rgba(249, 115, 22, 0.3)';
    ctx.fill();

    // Ic nokta
    ctx.beginPath();
    ctx.arc(lastX, lastY, 3, 0, Math.PI * 2);
    ctx.fillStyle = TOKENS.accent;
    ctx.fill();

    // X ekseni tarih etiketleri (ilk, orta, son)
    ctx.font = '10px ' + TOKENS.font;
    ctx.fillStyle = TOKENS.textMuted;
    ctx.textAlign = 'center';
    ctx.textBaseline = 'top';

    var labelIndices = [0, Math.floor(dataLen / 2), dataLen - 1];
    for (var idx = 0; idx < labelIndices.length; idx++) {
      var dataIdx = labelIndices[idx];
      if (dataIdx < dataLen) {
        var label = formatDateShort(priceHistory[dataIdx].date);
        ctx.fillText(label, xOf(dataIdx), padTop + chartH + 6);
      }
    }

    // Min/Max fiyat etiketleri (Y ekseni kenari)
    ctx.textAlign = 'left';
    ctx.textBaseline = 'middle';
    ctx.fillStyle = TOKENS.textMuted;
    ctx.font = '9px ' + TOKENS.font;

    // Max (ust)
    ctx.fillText(formatNumber(maxP, 0), padLeft + 2, padTop + 4);
    // Min (alt)
    ctx.fillText(formatNumber(minP, 0), padLeft + 2, padTop + chartH - 4);

    return canvas;
  }

  // ─── 7. Niche Score Badge ──────────────────────────────────

  /**
   * Nis skoru gosteren dairesel ilerleme rozeti.
   *
   * @param {Object} nicheData - Nis verileri
   * @param {number} nicheData.score - Skor (0-10)
   * @param {string} nicheData.verdict - Degerlendirme metni (orn: 'Girilmesi Onerilen')
   * @returns {HTMLElement} Rozet host elemani
   */
  function createNicheScoreBadge(nicheData) {
    if (!nicheData || typeof nicheData !== 'object') {
      return createSimpleElement('niche-badge-error');
    }

    var score = Number(nicheData.score) || 0;
    var verdict = nicheData.verdict || '';
    var color = getScoreColor(score);

    var css = [
      '.pr-niche-badge {',
      '  display: flex;',
      '  align-items: center;',
      '  gap: 12px;',
      '  padding: 12px 16px;',
      '}',
      '.pr-niche-circle {',
      '  position: relative;',
      '  width: 64px;',
      '  height: 64px;',
      '  flex-shrink: 0;',
      '}',
      '.pr-niche-svg {',
      '  width: 64px;',
      '  height: 64px;',
      '  transform: rotate(-90deg);',
      '}',
      '.pr-niche-text {',
      '  position: absolute;',
      '  top: 50%;',
      '  left: 50%;',
      '  transform: translate(-50%, -50%);',
      '  text-align: center;',
      '}',
      '.pr-niche-score {',
      '  font-size: 18px;',
      '  font-weight: 800;',
      '  line-height: 1;',
      '}',
      '.pr-niche-max {',
      '  font-size: 10px;',
      '  color: var(--pr-text-muted);',
      '  line-height: 1.4;',
      '}',
      '.pr-niche-info {',
      '  display: flex;',
      '  flex-direction: column;',
      '  gap: 2px;',
      '}',
      '.pr-niche-verdict {',
      '  font-size: 13px;',
      '  font-weight: 600;',
      '  color: var(--pr-text);',
      '  line-height: 1.3;',
      '}',
      '.pr-niche-hint {',
      '  font-size: 11px;',
      '  color: var(--pr-text-muted);',
      '}'
    ].join('\n');

    var parts = createShadowHost('niche-badge', css);
    var badge = el('div', { className: 'pr-container pr-glass pr-niche-badge' });

    // SVG dairesel ilerleme cubugu
    var radius = 26;
    var circumference = 2 * Math.PI * radius;
    var progress = Math.max(0, Math.min(score / 5, 1));
    var dashOffset = circumference * (1 - progress);

    var circleDiv = el('div', { className: 'pr-niche-circle' });

    // SVG - programatik olusturulmus (XSS guvenli)
    var svg = svgEl('svg', { class: 'pr-niche-svg', viewBox: '0 0 64 64' });

    var trackCircle = svgEl('circle', {
      cx: '32',
      cy: '32',
      r: String(radius),
      fill: 'none',
      stroke: 'rgba(255,255,255,0.08)',
      'stroke-width': '4'
    });

    var progressCircle = svgEl('circle', {
      cx: '32',
      cy: '32',
      r: String(radius),
      fill: 'none',
      stroke: color,
      'stroke-width': '4',
      'stroke-linecap': 'round',
      'stroke-dasharray': String(circumference),
      'stroke-dashoffset': String(dashOffset)
    });

    svg.appendChild(trackCircle);
    svg.appendChild(progressCircle);
    circleDiv.appendChild(svg);

    // Metin overlay
    var textDiv = el('div', { className: 'pr-niche-text' });
    textDiv.appendChild(el('div', {
      className: 'pr-niche-score',
      style: 'color:' + color
    }, score.toFixed(1)));
    textDiv.appendChild(el('div', { className: 'pr-niche-max' }, '/5'));
    circleDiv.appendChild(textDiv);

    // Degerlendirme metni
    var infoDiv = el('div', { className: 'pr-niche-info' });

    var hintMap = {
      high: 'Rekabet dusuk, talep yuksek',
      medium: 'Orta seviye firsat',
      low: 'Yuksek rekabet, dikkatli girilmeli'
    };
    var hintKey = score >= 7 ? 'high' : (score >= 4 ? 'medium' : 'low');

    infoDiv.appendChild(el('div', { className: 'pr-niche-verdict' }, verdict));
    infoDiv.appendChild(el('div', { className: 'pr-niche-hint' }, hintMap[hintKey]));

    badge.appendChild(circleDiv);
    badge.appendChild(infoDiv);

    parts.shadow.appendChild(badge);
    fadeIn(parts.host);
    return parts.host;
  }

  // ─── 8. Platform Comparison Table ──────────────────────────

  /**
   * 3 platformu yan yana karsilastiran tablo widget'i.
   *
   * @param {Object} comparisonData - Karsilastirma verisi (ProfitCalculator.compareAll sonucu)
   * @param {Object} comparisonData.trendyol - Trendyol hesaplama sonucu
   * @param {Object} comparisonData.hepsiburada - Hepsiburada hesaplama sonucu
   * @param {Object} comparisonData.n11 - N11 hesaplama sonucu
   * @param {string} comparisonData.best - En karli platform
   * @returns {HTMLElement} Tablo host elemani
   */
  function createPlatformComparison(comparisonData) {
    if (!comparisonData || typeof comparisonData !== 'object') {
      return createSimpleElement('comparison-error');
    }

    var css = [
      '.pr-comp-widget {',
      '  padding: 12px;',
      '  width: 100%;',
      '}',
      '.pr-comp-table {',
      '  width: 100%;',
      '  border-collapse: collapse;',
      '}',
      '.pr-comp-table th, .pr-comp-table td {',
      '  text-align: center;',
      '  padding: 6px 8px;',
      '  font-size: 12px;',
      '  border-bottom: 1px solid var(--pr-border);',
      '}',
      '.pr-comp-table th {',
      '  font-weight: 700;',
      '  color: var(--pr-text-secondary);',
      '  font-size: 11px;',
      '  text-transform: uppercase;',
      '  letter-spacing: 0.3px;',
      '}',
      '.pr-comp-best {',
      '  border: 1px solid ' + TOKENS.success + ' !important;',
      '  border-radius: 6px;',
      '  background: rgba(34, 197, 94, 0.08);',
      '}',
      '.pr-comp-star { color: ' + TOKENS.accentLight + '; margin-left: 2px; }',
      '.pr-comp-revenue { font-weight: 700; font-size: 14px; }',
      '.pr-comp-margin { font-weight: 600; }'
    ].join('\n');

    var parts = createShadowHost('comparison', css);
    var widget = el('div', { className: 'pr-container pr-glass pr-comp-widget' });

    var table = buildComparisonTable(comparisonData);
    widget.appendChild(table);

    parts.shadow.appendChild(widget);
    fadeIn(parts.host);
    return parts.host;
  }

  /**
   * Platform karsilastirma tablosunu DOM olarak olusturur.
   * Shadow DOM icerisinde kullanim icin tasarlanmistir.
   * @param {Object} comparisonData - Karsilastirma verisi
   * @returns {HTMLElement} Table elemani
   */
  function buildComparisonTable(comparisonData, isPro) {
    var platforms = [
      { key: 'trendyol', label: 'Trendyol' },
      { key: 'hepsiburada', label: 'Hepsibr.' },
      { key: 'n11', label: 'N11' }
    ];

    var best = comparisonData.best || '';
    // FREE kullanicilar icin: ilk platform acik, 2. ve 3. blur
    var shouldBlur = !isPro;

    var table = el('table', { className: 'pr-comp-table' });

    // Baslik satiri (platform isimleri her zaman gorunur)
    var thead = document.createElement('thead');
    var headerRow = document.createElement('tr');

    for (var h = 0; h < platforms.length; h++) {
      var th = document.createElement('th');
      th.appendChild(document.createTextNode(platforms[h].label));
      if (platforms[h].key === best) {
        th.appendChild(el('span', { className: 'pr-comp-star' }, '\u2605'));
      }
      headerRow.appendChild(th);
    }
    thead.appendChild(headerRow);
    table.appendChild(thead);

    // Veri satirlari
    var tbody = document.createElement('tbody');

    // Net Gelir satiri
    var revenueRow = document.createElement('tr');
    for (var r = 0; r < platforms.length; r++) {
      var td = document.createElement('td');
      td.className = 'pr-comp-revenue';
      var pData = comparisonData[platforms[r].key];
      if (pData) {
        td.textContent = formatCurrency(pData.netRevenue, 0);
        td.style.color = (platforms[r].key === best) ? TOKENS.success : TOKENS.text;
      } else {
        td.textContent = '-';
        td.style.color = TOKENS.textMuted;
      }
      if (platforms[r].key === best) {
        td.classList.add('pr-comp-best');
      }
      // 2. ve 3. platform blur
      if (shouldBlur && r > 0) {
        td.classList.add('pr-blur-value');
      }
      revenueRow.appendChild(td);
    }
    tbody.appendChild(revenueRow);

    // Marj satiri
    var marginRow = document.createElement('tr');
    for (var m = 0; m < platforms.length; m++) {
      var mtd = document.createElement('td');
      mtd.className = 'pr-comp-margin';
      var mData = comparisonData[platforms[m].key];

      if (mData && mData.netMarginPercent != null) {
        mtd.textContent = formatPercent(mData.netMarginPercent);
        mtd.style.color = getMarginColor(mData.netMarginPercent);
        if (platforms[m].key === best) {
          mtd.appendChild(el('span', { className: 'pr-comp-star' }, '\u2605'));
        }
      } else if (mData) {
        var totalCostRate = mData.salePrice > 0
          ? ((mData.totalMarketplaceCosts / mData.salePrice) * 100)
          : 0;
        mtd.textContent = formatPercent(100 - totalCostRate) + ' (gross)';
        mtd.style.color = TOKENS.textSecondary;
      } else {
        mtd.textContent = '-';
        mtd.style.color = TOKENS.textMuted;
      }
      if (shouldBlur && m > 0) {
        mtd.classList.add('pr-blur-value');
      }
      marginRow.appendChild(mtd);
    }
    tbody.appendChild(marginRow);

    // Komisyon satiri
    var commRow = document.createElement('tr');
    for (var c = 0; c < platforms.length; c++) {
      var ctd = document.createElement('td');
      var cData = comparisonData[platforms[c].key];
      if (cData) {
        ctd.textContent = formatPercent(cData.commissionRate);
        ctd.style.color = TOKENS.textSecondary;
      } else {
        ctd.textContent = '-';
        ctd.style.color = TOKENS.textMuted;
      }
      if (shouldBlur && c > 0) {
        ctd.classList.add('pr-blur-value');
      }
      commRow.appendChild(ctd);
    }
    tbody.appendChild(commRow);

    table.appendChild(tbody);
    return table;
  }

  // ─── 9. Upgrade Prompt ─────────────────────────────────────

  /**
   * Ucretsiz kullanicilara gosterilen PRO yukseltme istemi.
   * Gunluk limit dolunca tetiklenir.
   *
   * @returns {HTMLElement} Prompt host elemani
   */
  function createUpgradePrompt() {
    var css = [
      '.pr-upgrade-overlay {',
      '  position: fixed;',
      '  top: 0;',
      '  left: 0;',
      '  right: 0;',
      '  bottom: 0;',
      '  background: rgba(0,0,0,0.6);',
      '  display: flex;',
      '  align-items: center;',
      '  justify-content: center;',
      '  z-index: ' + (Z_INDEX + 1) + ';',
      '}',
      '.pr-upgrade-card {',
      '  width: 340px;',
      '  padding: 24px;',
      '  text-align: center;',
      '  position: relative;',
      '}',
      '.pr-upgrade-icon {',
      '  font-size: 36px;',
      '  margin-bottom: 12px;',
      '}',
      '.pr-upgrade-title {',
      '  font-size: 18px;',
      '  font-weight: 700;',
      '  color: var(--pr-text);',
      '  margin-bottom: 16px;',
      '}',
      '.pr-upgrade-features {',
      '  text-align: left;',
      '  margin-bottom: 20px;',
      '}',
      '.pr-upgrade-feature {',
      '  display: flex;',
      '  align-items: center;',
      '  gap: 8px;',
      '  padding: 6px 0;',
      '  font-size: 13px;',
      '  color: var(--pr-text-secondary);',
      '}',
      '.pr-upgrade-check {',
      '  color: ' + TOKENS.success + ';',
      '  font-size: 14px;',
      '  font-weight: 700;',
      '  flex-shrink: 0;',
      '}',
      '.pr-upgrade-btn {',
      '  width: 100%;',
      '  padding: 12px 20px;',
      '  font-size: 15px;',
      '  background: linear-gradient(135deg, ' + TOKENS.accent + ', ' + TOKENS.accentDark + ');',
      '  box-shadow: 0 4px 16px rgba(249,115,22,0.3);',
      '}',
      '.pr-upgrade-btn:hover {',
      '  box-shadow: 0 6px 24px rgba(249,115,22,0.45);',
      '}',
      '.pr-upgrade-price {',
      '  font-size: 12px;',
      '  color: var(--pr-text-muted);',
      '  margin-top: 10px;',
      '}',
      '.pr-upgrade-close {',
      '  position: absolute;',
      '  top: 12px;',
      '  right: 12px;',
      '}'
    ].join('\n');

    var parts = createShadowHost('upgrade-prompt', css);

    var overlay = el('div', { className: 'pr-upgrade-overlay' });
    var card = el('div', { className: 'pr-container pr-glass pr-upgrade-card' });

    // Kapatma butonu
    var closeBtn = el('button', { className: 'pr-close-btn pr-upgrade-close' }, '\u2715');
    closeBtn.addEventListener('click', function () {
      fadeOut(parts.host);
    });
    card.appendChild(closeBtn);

    // Icon
    card.appendChild(el('div', { className: 'pr-upgrade-icon' }, '\uD83D\uDD12'));

    // Baslik
    card.appendChild(el('div', { className: 'pr-upgrade-title' }, 'Gunluk Limit Doldu'));

    // Ozellik listesi
    var features = [
      'Sinirsiz urun analizi',
      '3 platform karsilastirma',
      'Satis tahmini ve trend analizi',
      'Fiyat gecmisi grafikleri',
      'Nis analizi ve puanlama',
      'Toplu urun analizi (CSV)'
    ];

    var featuresDiv = el('div', { className: 'pr-upgrade-features' });
    for (var i = 0; i < features.length; i++) {
      var featureRow = el('div', { className: 'pr-upgrade-feature' });
      featureRow.appendChild(el('span', { className: 'pr-upgrade-check' }, '\u2713'));
      featureRow.appendChild(el('span', null, features[i]));
      featuresDiv.appendChild(featureRow);
    }
    card.appendChild(featuresDiv);

    // Yukseltme butonu
    var upgradeBtn = el('button', { className: 'pr-btn pr-upgrade-btn' }, 'PRO\'ya Yukselt');
    upgradeBtn.addEventListener('click', function () {
      var event = new CustomEvent('pazarradar:upgrade-click', { bubbles: true });
      parts.host.dispatchEvent(event);
    });
    card.appendChild(upgradeBtn);

    // Fiyat notu
    card.appendChild(el('div', { className: 'pr-upgrade-price' }, '149 TL/ay - 7 gun ucretsiz deneyin'));

    // Arka plana tiklaninca kapat
    overlay.addEventListener('click', function (e) {
      if (e.target === overlay) {
        fadeOut(parts.host);
      }
    });

    overlay.appendChild(card);
    parts.shadow.appendChild(overlay);
    fadeIn(parts.host);
    return parts.host;
  }

  // ─── 10. Toast Notification ────────────────────────────────

  /**
   * Kisa sureli bildirim mesaji gosterir.
   *
   * @param {string} message - Bildirim metni
   * @param {string} [type='info'] - Bildirim tipi ('info'|'success'|'error'|'warning')
   */
  function showToast(message, type) {
    var toastType = type || 'info';
    var colorMap = {
      info: TOKENS.accent,
      success: TOKENS.success,
      error: TOKENS.danger,
      warning: TOKENS.warning
    };
    var iconMap = {
      info: '\u2139',
      success: '\u2713',
      error: '\u2717',
      warning: '\u26A0'
    };

    var accentColor = colorMap[toastType] || colorMap.info;
    var icon = iconMap[toastType] || iconMap.info;

    var css = [
      '.pr-toast {',
      '  position: fixed;',
      '  right: 20px;',
      '  z-index: ' + (Z_INDEX + 2) + ';',
      '  display: flex;',
      '  align-items: center;',
      '  gap: 10px;',
      '  padding: 12px 18px;',
      '  max-width: 360px;',
      '  min-width: 200px;',
      '  pointer-events: auto;',
      '}',
      '.pr-toast-icon {',
      '  flex-shrink: 0;',
      '  width: 24px;',
      '  height: 24px;',
      '  border-radius: 50%;',
      '  display: flex;',
      '  align-items: center;',
      '  justify-content: center;',
      '  font-size: 14px;',
      '  font-weight: 700;',
      '  color: #fff;',
      '}',
      '.pr-toast-msg {',
      '  font-size: 13px;',
      '  color: var(--pr-text);',
      '  line-height: 1.4;',
      '}',
      '.pr-toast-close {',
      '  flex-shrink: 0;',
      '  margin-left: auto;',
      '}'
    ].join('\n');

    var parts = createShadowHost('toast', css);

    // Mevcut toast'lari yukariya kaydir
    var existingCount = activeToasts.length;
    var bottomOffset = 80 + (existingCount * 60);

    var toast = el('div', {
      className: 'pr-container pr-glass pr-toast',
      style: 'bottom:' + bottomOffset + 'px;'
    });

    var iconEl = el('div', {
      className: 'pr-toast-icon',
      style: 'background:' + accentColor
    }, icon);

    var msgEl = el('div', { className: 'pr-toast-msg' }, message);

    var closeBtn = el('button', { className: 'pr-close-btn pr-toast-close' }, '\u2715');

    var timerId;
    var removeToast = function () {
      clearTimeout(timerId);
      var idx = activeToasts.indexOf(parts.host);
      if (idx > -1) {
        activeToasts.splice(idx, 1);
      }
      fadeOut(parts.host);
    };

    closeBtn.addEventListener('click', removeToast);

    toast.appendChild(iconEl);
    toast.appendChild(msgEl);
    toast.appendChild(closeBtn);

    parts.shadow.appendChild(toast);

    activeToasts.push(parts.host);
    document.body.appendChild(parts.host);
    fadeIn(parts.host);

    // Otomatik kapanma
    timerId = setTimeout(removeToast, TOAST_AUTO_CLOSE);

    // Uzerine gelinince zamanlayiciyi durdur
    toast.addEventListener('mouseenter', function () {
      clearTimeout(timerId);
    });
    toast.addEventListener('mouseleave', function () {
      timerId = setTimeout(removeToast, TOAST_AUTO_CLOSE);
    });
  }

  // ─── 11. Cleanup ──────────────────────────────────────────

  /**
   * Tum enjekte edilmis PazarRadar elemanlarini DOM'dan kaldirir.
   */
  function cleanup() {
    // Takip edilen elemanlari kaldir
    for (var i = injectedElements.length - 1; i >= 0; i--) {
      var elem = injectedElements[i];
      if (elem && elem.parentNode) {
        elem.parentNode.removeChild(elem);
      }
    }
    injectedElements.length = 0;

    // Aktif toast'lari temizle
    for (var t = activeToasts.length - 1; t >= 0; t--) {
      var toast = activeToasts[t];
      if (toast && toast.parentNode) {
        toast.parentNode.removeChild(toast);
      }
    }
    activeToasts.length = 0;

    // Guvenceli tarama: DOM'da kalan data-pazarradar elemanlarini da temizle
    var remaining = document.querySelectorAll('[' + DATA_ATTR + ']');
    for (var r = 0; r < remaining.length; r++) {
      if (remaining[r].parentNode) {
        remaining[r].parentNode.removeChild(remaining[r]);
      }
    }
  }

  // ─── Public API ─────────────────────────────────────────────

  window.PazarRadar.InjectUI = Object.freeze({

    /**
     * Tum sayfalarda gorunen PazarRadar floating badge olusturur.
     * @returns {HTMLElement}
     */
    createBadge: createBadge,

    /**
     * Arama sonuc karti olusturur (her urun listesinin altina enjekte edilir).
     * @param {Object} data - Kart verileri
     * @returns {HTMLElement}
     */
    createSearchResultCard: createSearchResultCard,

    /**
     * Arama sonuclari sayfasinin ustune yerlestirilen toplu analiz cubugu.
     * @param {Object} data - Toplu veri
     * @returns {HTMLElement}
     */
    createSearchAggregateBar: createSearchAggregateBar,

    /**
     * Urun detay sayfasinda gorunen sag panel.
     * @param {Object} data - Panel verileri
     * @returns {HTMLElement}
     */
    createProductPanel: createProductPanel,

    /**
     * Kompakt kar hesaplayici widget.
     * @param {Object} data - Hesaplayici verisi
     * @returns {HTMLElement}
     */
    createProfitMiniCalc: createProfitMiniCalc,

    /**
     * Fiyat gecmisi grafik widget'i.
     * @param {Array<Object>} priceHistory - Fiyat verileri
     * @returns {HTMLElement}
     */
    createPriceChart: createPriceChart,

    /**
     * Nis skoru dairesel rozet widget'i.
     * @param {Object} nicheData - Nis verileri
     * @returns {HTMLElement}
     */
    createNicheScoreBadge: createNicheScoreBadge,

    /**
     * Platform karsilastirma tablosu widget'i.
     * @param {Object} comparisonData - Karsilastirma verisi
     * @returns {HTMLElement}
     */
    createPlatformComparison: createPlatformComparison,

    /**
     * PRO yukseltme istemi overlay'i.
     * @returns {HTMLElement}
     */
    createUpgradePrompt: createUpgradePrompt,

    /**
     * Kisa sureli bildirim mesaji gosterir.
     * @param {string} message - Bildirim metni
     * @param {string} [type='info'] - Bildirim tipi
     */
    showToast: showToast,

    /**
     * Tum enjekte edilmis elemanlari temizler.
     */
    cleanup: cleanup,

    /**
     * Yardimci formatlama fonksiyonlari (content script'ler icin kullanilabilir).
     */
    utils: Object.freeze({
      formatNumber: formatNumber,
      formatCompact: formatCompact,
      formatCurrency: formatCurrency,
      formatPercent: formatPercent,
      formatDateShort: formatDateShort,
      getMarginColor: getMarginColor,
      getScoreColor: getScoreColor,
      getTrendInfo: getTrendInfo
    })
  });
})();
