/**
 * PazarRadar - N11 Content Script
 *
 * N11.com uzerinde calisan icerik betigi. Urun arama sonuclari,
 * urun detay sayfasi ve magaza sayfalari uzerinden veri cikarip
 * PazarRadar analiz panellerini sayfaya enjekte eder.
 *
 * Bagimliliklar:
 *   - window.PazarRadar.CommissionDB
 *   - window.PazarRadar.ProfitCalculator
 *   - window.PazarRadar.SalesEstimator
 *   - window.PazarRadar.InjectUI
 *
 * @namespace N11ContentScript
 */
(function () {
  'use strict';

  // ─── Global Erisim & Guvensizlik Korumalari ────────────────
  var PR;
  try {
    PR = window.PazarRadar;
  } catch (_err) {
    return;
  }

  if (!PR) {
    return;
  }

  var PLATFORM = 'n11';
  var LOG_PREFIX = '[PazarRadar][N11]';

  /** Production modunda konsol ciktisini devre disi birakir */
  var DEBUG = false;

  /** Lisans durumu - chrome.storage'dan yuklenir */
  var userIsPro = false;
  var userTrialDaysLeft = 0;

  // Lisans durumunu background'dan oku
  try {
    if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.local) {
      chrome.storage.local.get('license', function (result) {
        var license = result && result.license;
        if (license && license.plan && license.plan !== 'free') {
          if (!license.validUntil || Date.now() <= license.validUntil) {
            userIsPro = true;
          }
          // Trial gun hesabi
          if (license.plan === 'trial' && license.trialEndsAt) {
            var msLeft = license.trialEndsAt - Date.now();
            userTrialDaysLeft = Math.max(0, Math.ceil(msLeft / (24 * 60 * 60 * 1000)));
            if (userTrialDaysLeft > 0) { userIsPro = true; }
          }
        }
      });
    }
  } catch (_e) { /* Content script ortaminda storage erisimi olmayabilir */ }

  // ─── Sabitler ───────────────────────────────────────────────

  /** Sayfa tipi sabitleri */
  var PAGE_TYPES = Object.freeze({
    SEARCH: 'search',
    PRODUCT: 'product',
    STORE: 'store',
    UNKNOWN: 'unknown'
  });

  /**
   * CSS secici havuzu. N11'in farkli sayfa versiyonlarinda calismasi icin
   * birden fazla secici tanimlanir; ilk eslesen kullanilir.
   */
  var SELECTORS = Object.freeze({
    // Arama sonuclari
    searchResultContainer: [
      '.resultListContainer',
      '.searchResultList',
      '.listingListContent',
      '#productListContainer',
      '.product-list-holder'
    ],
    productCard: [
      '.columnContent',
      '.product-card',
      '.pro',
      '.listingList li',
      '.product-list-item',
      '.search-result-item'
    ],
    productTitle: [
      '.productName',
      '.pro-name',
      '.product-title',
      'h3.pro-name a',
      '.product-card-title'
    ],
    productPrice: [
      '.newPrice',
      '.pro-price .price',
      '.price-sales',
      '.product-price .newPrice',
      '.priceContainer .newPrice ins'
    ],
    productOriginalPrice: [
      '.oldPrice',
      '.pro-price .oldPrice',
      '.price-standard',
      '.priceContainer .oldPrice del'
    ],
    productLink: [
      'a[href*="/urun/"]',
      'a.productLink',
      'a.pro-link',
      '.productName a',
      '.pro-name a'
    ],
    productRating: [
      '.ratingCont',
      '.rating',
      '.pro-rating',
      '.product-rating',
      '.star-rating'
    ],
    productReviewCount: [
      '.ratingText',
      '.review-count',
      '.pro-review-count',
      '.rating-count'
    ],
    productSeller: [
      '.sallerName',
      '.sellerName',
      '.seller-name',
      '.pro-seller',
      'a[href*="/magaza/"]'
    ],

    // Urun detay sayfasi
    detailTitle: [
      '.proName',
      '.titleArea h1',
      'h1.title',
      'h1.product-name',
      'h1.pro-name',
      '#product-name',
      '.product-detail-name h1',
      '.unf-p-name'
    ],
    detailPrice: [
      '.newPrice',
      '.newPrice ins',
      '.price-sales',
      '.product-price .newPrice',
      '.unf-p-price .newPrice',
      '#offering-price .price',
      '.priceContainer .newPrice',
      '.prc-dsc',
      '.sale-price',
      '[data-price]'
    ],
    detailOriginalPrice: [
      '.oldPrice',
      '.price-standard',
      '.product-price .oldPrice',
      '.unf-p-price .oldPrice'
    ],
    detailRating: [
      '.ratingCont',
      '.comment-star',
      '.product-rating',
      '.rating-line',
      '#product-rating',
      '.unf-p-rating'
    ],
    detailReviewCount: [
      '.ratingText',
      'a[href*="product-reviews"]',
      '.review-count',
      '.total-review-count',
      '#productReviewCount',
      '.unf-p-review-count'
    ],
    detailSeller: [
      '.sallerName',
      '.sidebarSellerArea-top-name',
      'a.sidebarSellerArea-top-name',
      '.seller-name-text',
      '.sellerName',
      '.seller-info .name',
      'a.sallerName',
      '.unf-p-seller-name'
    ],
    detailBreadcrumb: [
      '.breadcrumb',
      '.breadcrumb-list',
      'nav[aria-label="breadcrumb"]',
      '.categoryPath',
      '#breadcrumb',
      '.unf-breadcrumb'
    ],
    detailSpecTable: [
      '.data-list',
      '.product-feature-group',
      '.spec-table',
      '.productFeatureGroup',
      '#tabProductDesc .data-list',
      '.unf-p-spec'
    ],

    // Magaza sayfasi
    storeName: [
      '.store-name',
      '.shopName',
      '.seller-name',
      'h1.store-title',
      '.merchant-name'
    ],
    storeProducts: [
      '.columnContent',
      '.product-card',
      '.pro',
      '.store-product-item'
    ]
  });

  /** N11 URL desenleri */
  var URL_PATTERNS = Object.freeze({
    search: /n11\.com\/(arama\?|kategori\/|.*[?&]q=)/i,
    product: /n11\.com\/urun\//i,
    store: /n11\.com\/magaza\//i
  });

  /** Urun ID regex deseni - n11 URL'den numara cikartir */
  var PRODUCT_ID_REGEX = /\/urun\/.*?-(\d+)(?:\?|$|\/)/i;

  /** Alternatif ID deseni - sadece sondaki sayi blogu */
  var PRODUCT_ID_FALLBACK_REGEX = /-(\d{5,})(?:\?|$|\/)/i;

  /** Enjeksiyon isareti - birden fazla calismayi onle */
  var INJECTED_ATTR = 'data-pazar-radar-injected';

  /** SPA gecislerinde yeniden enjeksiyon icin bekleme suresi (ms) */
  var SPA_REINJECT_DELAY = 800;

  /** MutationObserver debounce suresi (ms) */
  var MUTATION_DEBOUNCE_MS = 400;

  // ─── Yardimci Fonksiyonlar ──────────────────────────────────

  /**
   * Birden fazla secici arasinda ilk eslesen elementi bulur.
   * @param {Element|Document} context - Arama yapilacak ust element
   * @param {string[]} selectorList - Denenecek CSS seciciler
   * @returns {Element|null}
   */
  function queryFirst(context, selectorList) {
    if (!context || !selectorList) return null;

    for (var i = 0; i < selectorList.length; i++) {
      try {
        var el = context.querySelector(selectorList[i]);
        if (el) return el;
      } catch (_e) {
        // Gecersiz secici - bir sonrakini dene
      }
    }
    return null;
  }

  /**
   * Birden fazla secici arasinda ilk eslesen tum elementleri bulur.
   * @param {Element|Document} context - Arama yapilacak ust element
   * @param {string[]} selectorList - Denenecek CSS seciciler
   * @returns {Element[]}
   */
  function queryAllFirst(context, selectorList) {
    if (!context || !selectorList) return [];

    for (var i = 0; i < selectorList.length; i++) {
      try {
        var els = context.querySelectorAll(selectorList[i]);
        if (els && els.length > 0) return Array.from(els);
      } catch (_e) {
        // Gecersiz secici - bir sonrakini dene
      }
    }
    return [];
  }

  /**
   * Element metin icerigini temizleyerek dondurur.
   * @param {Element|null} el
   * @returns {string}
   */
  function getTextContent(el) {
    if (!el) return '';
    return (el.textContent || '').replace(/\s+/g, ' ').trim();
  }

  /**
   * Fiyat metninden sayisal deger cikarir.
   * "1.299,90 TL" => 1299.90
   * @param {string} priceText
   * @returns {number}
   */
  function parsePrice(priceText) {
    if (!priceText) return 0;

    var cleaned = priceText
      .replace(/[^\d.,]/g, '')  // Sadece rakam, nokta, virgul birak
      .replace(/\./g, '')       // Binlik ayiraci kaldir
      .replace(',', '.');       // Ondalik virgulu noktaya cevir

    var value = parseFloat(cleaned);
    return Number.isFinite(value) && value > 0 ? value : 0;
  }

  /**
   * Yildiz puanini cikarir. Metin, aria-label veya CSS width'ten hesaplar.
   * @param {Element|null} ratingEl
   * @returns {number}
   */
  function parseRating(ratingEl) {
    if (!ratingEl) return 0;

    // Metin iceriginden dene
    var text = getTextContent(ratingEl);
    var match = text.match(/([\d]+[.,]?\d*)/);
    if (match) {
      var val = parseFloat(match[1].replace(',', '.'));
      if (Number.isFinite(val) && val >= 0 && val <= 5) return val;
    }

    // aria-label'dan dene
    var ariaLabel = ratingEl.getAttribute('aria-label') || '';
    var ariaMatch = ariaLabel.match(/([\d]+[.,]?\d*)/);
    if (ariaMatch) {
      var ariaVal = parseFloat(ariaMatch[1].replace(',', '.'));
      if (Number.isFinite(ariaVal) && ariaVal >= 0 && ariaVal <= 5) return ariaVal;
    }

    // N11'de yildiz genisligi ile puan hesaplama
    var starFill = ratingEl.querySelector(
      '.star-fill, .ratingPoint, .full, [style*="width"]'
    );
    if (starFill) {
      var style = starFill.getAttribute('style') || '';
      var widthMatch = style.match(/width:\s*([\d.]+)%/);
      if (widthMatch) {
        return Math.round((parseFloat(widthMatch[1]) / 20) * 10) / 10; // %100 = 5 yildiz
      }
    }

    // data-score attribute kontrolu
    var score = ratingEl.getAttribute('data-score') || ratingEl.getAttribute('data-rating');
    if (score) {
      var scoreVal = parseFloat(score);
      if (Number.isFinite(scoreVal) && scoreVal >= 0 && scoreVal <= 5) return scoreVal;
    }

    return 0;
  }

  /**
   * Yorum/degerlendirme sayisini cikarir.
   * "(245)" veya "245 degerlendirme" => 245
   * @param {Element|null} countEl
   * @returns {number}
   */
  function parseReviewCount(countEl) {
    if (!countEl) return 0;

    var text = getTextContent(countEl);
    var match = text.match(/([\d.]+)/);
    if (match) {
      var val = parseInt(match[1].replace(/\./g, ''), 10);
      return Number.isFinite(val) ? val : 0;
    }
    return 0;
  }

  /**
   * Breadcrumb'dan kategori bilgisini cikarir.
   * @param {Element|Document} context
   * @returns {string}
   */
  function extractCategoryFromBreadcrumb(context) {
    var breadcrumb = queryFirst(context, SELECTORS.detailBreadcrumb);
    if (!breadcrumb) return '';

    var items = breadcrumb.querySelectorAll('a, span, li');
    if (!items || items.length === 0) return '';

    // Son anlamli breadcrumb ogesi kategoridir (ilk oge genelde "Anasayfa")
    var categories = [];
    for (var i = 0; i < items.length; i++) {
      var text = getTextContent(items[i]);
      if (text &&
          text.toLowerCase() !== 'anasayfa' &&
          text.toLowerCase() !== 'n11' &&
          text !== '>' &&
          text !== '/') {
        categories.push(text);
      }
    }

    // Son kategoriyi dondur (en spesifik)
    return categories.length > 0 ? categories[categories.length - 1] : '';
  }

  /**
   * URL'den veya data attribute'den urun ID'sini cikarir.
   * @param {string} url
   * @param {Element} [card] - Urun karti elementi
   * @returns {string}
   */
  function extractProductId(url, card) {
    // data attribute'lardan dene
    if (card) {
      var dataId = card.getAttribute('data-productid') ||
                   card.getAttribute('data-product-id') ||
                   card.getAttribute('data-id') ||
                   card.getAttribute('data-sku');
      if (dataId) return dataId;
    }

    // URL'den cikar: /urun/urun-adi-123456
    if (url) {
      var match = url.match(PRODUCT_ID_REGEX);
      if (match) return match[1];

      // Fallback desen
      var fallback = url.match(PRODUCT_ID_FALLBACK_REGEX);
      if (fallback) return fallback[1];
    }

    return '';
  }

  /**
   * JSON-LD yapisal verisini sayfadan cikarir.
   * @returns {Object|null}
   */
  function extractJsonLd() {
    try {
      var scripts = document.querySelectorAll('script[type="application/ld+json"]');
      for (var i = 0; i < scripts.length; i++) {
        var data = JSON.parse(scripts[i].textContent || '{}');

        // Dizi icerisinde Product araniyor
        if (Array.isArray(data)) {
          for (var j = 0; j < data.length; j++) {
            if (data[j]['@type'] === 'Product') return data[j];
          }
        }

        // Direkt Product nesnesi
        if (data['@type'] === 'Product') return data;

        // @graph icinde arama
        if (data['@graph'] && Array.isArray(data['@graph'])) {
          for (var k = 0; k < data['@graph'].length; k++) {
            if (data['@graph'][k]['@type'] === 'Product') return data['@graph'][k];
          }
        }
      }
    } catch (_err) {
      warn('JSON-LD parse hatasi');
    }
    return null;
  }

  /**
   * Konsola uyari logu yazar.
   * @param {...*} args
   */
  function warn() {
    if (!DEBUG) return;
    var args = [LOG_PREFIX].concat(Array.from(arguments));
    try { console.warn.apply(console, args); } catch (_e) { /* noop */ }
  }

  /**
   * Konsola bilgi logu yazar.
   * @param {...*} args
   */
  function info() {
    if (!DEBUG) return;
    var args = [LOG_PREFIX].concat(Array.from(arguments));
    try { console.info.apply(console, args); } catch (_e) { /* noop */ }
  }

  /**
   * Debounce fonksiyonu.
   * @param {Function} fn
   * @param {number} ms
   * @returns {Function}
   */
  function debounce(fn, ms) {
    var timer;
    return function () {
      var args = arguments;
      var self = this;
      clearTimeout(timer);
      timer = setTimeout(function () {
        fn.apply(self, args);
      }, ms);
    };
  }

  // ─── Sayfa Tipi Tespiti ─────────────────────────────────────

  /**
   * Mevcut URL'ye gore sayfa tipini tespit eder.
   * @returns {string} PAGE_TYPES degerlerinden biri
   */
  function detectPageType() {
    var url = location.href;

    if (URL_PATTERNS.product.test(url)) return PAGE_TYPES.PRODUCT;
    if (URL_PATTERNS.store.test(url)) return PAGE_TYPES.STORE;
    if (URL_PATTERNS.search.test(url)) return PAGE_TYPES.SEARCH;

    // URL deseniyle eslesmeyen ama urun listesi iceren sayfa olabilir
    var container = queryFirst(document, SELECTORS.searchResultContainer);
    if (container) return PAGE_TYPES.SEARCH;

    return PAGE_TYPES.UNKNOWN;
  }

  // ─── Veri Cikarma: Arama Sonuclari ─────────────────────────

  /**
   * Tek bir urun kartindan veri cikarir.
   * @param {Element} card - Urun karti DOM elementi
   * @returns {Object|null} Cikarilan urun verisi veya null
   */
  function extractProductFromCard(card) {
    try {
      if (!card) return null;

      var titleEl = queryFirst(card, SELECTORS.productTitle);
      var priceEl = queryFirst(card, SELECTORS.productPrice);
      var originalPriceEl = queryFirst(card, SELECTORS.productOriginalPrice);
      var linkEl = queryFirst(card, SELECTORS.productLink);
      var ratingEl = queryFirst(card, SELECTORS.productRating);
      var reviewCountEl = queryFirst(card, SELECTORS.productReviewCount);
      var sellerEl = queryFirst(card, SELECTORS.productSeller);

      var title = getTextContent(titleEl);
      var price = parsePrice(getTextContent(priceEl));

      // Baslik veya fiyat yoksa gecersiz kart
      if (!title && price <= 0) return null;

      var url = (linkEl && linkEl.href) ? linkEl.href : '';
      var productId = extractProductId(url, card);
      var seller = getTextContent(sellerEl);

      return {
        platform: PLATFORM,
        title: title,
        price: price,
        originalPrice: parsePrice(getTextContent(originalPriceEl)),
        rating: parseRating(ratingEl),
        reviewCount: parseReviewCount(reviewCountEl),
        seller: seller,
        url: url,
        productId: productId,
        category: extractCategoryFromBreadcrumb(document),
        favoriteCount: 0,
        cardElement: card
      };
    } catch (err) {
      warn('Kart verisi cikarma hatasi:', err);
      return null;
    }
  }

  /**
   * Arama sayfasindaki tum urun kartlarindan veri cikarir.
   * @returns {Object[]} Cikarilan urun listesi
   */
  function extractSearchProducts() {
    var cards = queryAllFirst(document, SELECTORS.productCard);

    if (cards.length === 0) {
      warn('Urun karti bulunamadi');
      return [];
    }

    var products = [];
    for (var i = 0; i < cards.length; i++) {
      var product = extractProductFromCard(cards[i]);
      if (product) {
        products.push(product);
      }
    }

    info('Toplam ' + products.length + ' urun cikarildi');
    return products;
  }

  // ─── Veri Cikarma: Urun Detay ──────────────────────────────

  /**
   * Urun ozellikler/spesifikasyon tablosunu cikarir.
   * @returns {Object} Anahtar-deger ciftleri
   */
  function extractSpecifications() {
    var specs = {};

    try {
      var specTable = queryFirst(document, SELECTORS.detailSpecTable);
      if (!specTable) return specs;

      // Tablo satirlari
      var rows = specTable.querySelectorAll('tr, .data-list-item, li');
      for (var i = 0; i < rows.length; i++) {
        var cells = rows[i].querySelectorAll('td, th, span, .attr-name, .attr-value');
        if (cells.length >= 2) {
          var key = getTextContent(cells[0]);
          var value = getTextContent(cells[1]);
          if (key) {
            specs[key] = value;
          }
        }
      }

      // dt/dd yapisi destegi
      var dtElements = specTable.querySelectorAll('dt');
      var ddElements = specTable.querySelectorAll('dd');
      var len = Math.min(dtElements.length, ddElements.length);
      for (var j = 0; j < len; j++) {
        var dtText = getTextContent(dtElements[j]);
        var ddText = getTextContent(ddElements[j]);
        if (dtText) {
          specs[dtText] = ddText;
        }
      }

      // N11 ozel format: "ozellik: deger" seklinde li elementleri
      var listItems = specTable.querySelectorAll('li, .feature-item');
      for (var k = 0; k < listItems.length; k++) {
        var itemText = getTextContent(listItems[k]);
        var colonIndex = itemText.indexOf(':');
        if (colonIndex > 0) {
          var featureKey = itemText.substring(0, colonIndex).trim();
          var featureValue = itemText.substring(colonIndex + 1).trim();
          if (featureKey && featureValue && !specs[featureKey]) {
            specs[featureKey] = featureValue;
          }
        }
      }
    } catch (err) {
      warn('Spesifikasyon cikarma hatasi:', err);
    }

    return specs;
  }

  /**
   * Urun detay sayfasindan tum verileri cikarir.
   * JSON-LD ve DOM elementlerini birlestirir.
   * @returns {Object|null} Urun detay verisi
   */
  function extractProductDetail() {
    try {
      // Once JSON-LD'den zengin veri almaya calis
      var jsonLd = extractJsonLd();

      // DOM'dan cikarma
      var titleEl = queryFirst(document, SELECTORS.detailTitle);
      var priceEl = queryFirst(document, SELECTORS.detailPrice);
      var originalPriceEl = queryFirst(document, SELECTORS.detailOriginalPrice);
      var ratingEl = queryFirst(document, SELECTORS.detailRating);
      var reviewCountEl = queryFirst(document, SELECTORS.detailReviewCount);
      var sellerEl = queryFirst(document, SELECTORS.detailSeller);

      var title = getTextContent(titleEl);
      var price = parsePrice(getTextContent(priceEl));

      // data-price attribute fallback
      if (price <= 0) {
        var dataPriceEl = document.querySelector('[data-price]');
        if (dataPriceEl) {
          price = parseFloat(dataPriceEl.getAttribute('data-price')) || 0;
        }
      }

      // JSON-LD varsa DOM verisini zenginlestir
      if (jsonLd) {
        if (!title && jsonLd.name) {
          title = jsonLd.name;
        }
        if (price <= 0 && jsonLd.offers) {
          var offers = Array.isArray(jsonLd.offers) ? jsonLd.offers[0] : jsonLd.offers;
          if (offers && offers.price) {
            price = parseFloat(offers.price) || 0;
          }
        }
      }

      if (!title && price <= 0) {
        warn('Urun detayi cikarilacak veri bulunamadi');
        return null;
      }

      // Fiyat bulunamadiysa (Vue.js gecikme) null don ki retry tetiklensin
      if (price <= 0) {
        warn('Fiyat henuz yuklenemedi, retry beklenecek');
        return null;
      }

      var rating = parseRating(ratingEl);
      var reviewCount = parseReviewCount(reviewCountEl);

      // JSON-LD'den rating/review zenginlestirmesi
      if (jsonLd && jsonLd.aggregateRating) {
        if (rating <= 0 && jsonLd.aggregateRating.ratingValue) {
          rating = parseFloat(jsonLd.aggregateRating.ratingValue) || 0;
        }
        if (reviewCount <= 0 && jsonLd.aggregateRating.reviewCount) {
          reviewCount = parseInt(jsonLd.aggregateRating.reviewCount, 10) || 0;
        }
      }

      var productId = extractProductId(location.href, null);
      var category = extractCategoryFromBreadcrumb(document);
      var seller = getTextContent(sellerEl);
      var specifications = extractSpecifications();

      return {
        platform: PLATFORM,
        title: title,
        price: price,
        originalPrice: parsePrice(getTextContent(originalPriceEl)),
        rating: rating,
        reviewCount: reviewCount,
        seller: seller,
        url: location.href,
        productId: productId,
        category: category,
        favoriteCount: 0,
        specifications: specifications
      };
    } catch (err) {
      warn('Urun detay cikarma hatasi:', err);
      return null;
    }
  }

  // ─── Veri Cikarma: Magaza Sayfasi ──────────────────────────

  /**
   * Magaza sayfasindan satici ve urun bilgilerini cikarir.
   * @returns {Object|null}
   */
  function extractStoreData() {
    try {
      var storeNameEl = queryFirst(document, SELECTORS.storeName);
      var storeName = getTextContent(storeNameEl);

      var productCards = queryAllFirst(document, SELECTORS.storeProducts);
      var products = [];

      for (var i = 0; i < productCards.length; i++) {
        var product = extractProductFromCard(productCards[i]);
        if (product) {
          products.push(product);
        }
      }

      return {
        platform: PLATFORM,
        storeName: storeName,
        storeUrl: location.href,
        productCount: products.length,
        products: products
      };
    } catch (err) {
      warn('Magaza veri cikarma hatasi:', err);
      return null;
    }
  }

  // ─── Analiz Motoru ──────────────────────────────────────────

  /**
   * Cikarilan urun verisini PazarRadar motorlarina gonderir ve analiz sonucu dondurur.
   * @param {Object} data - Cikarilan urun verisi
   * @returns {Object} Zenginlestirilmis analiz sonucu
   */
  function analyzeProduct(data) {
    if (!data) return data;

    var result = {
      platform: PLATFORM,
      title: data.title,
      price: data.price,
      originalPrice: data.originalPrice,
      rating: data.rating,
      reviewCount: data.reviewCount,
      seller: data.seller,
      url: data.url,
      productId: data.productId,
      category: data.category,
      favoriteCount: data.favoriteCount || 0,
      profit: null,
      sales: null,
      comparison: null,
      categoryInfo: null,
      specifications: data.specifications || null
    };

    // Kategori bilgisi
    try {
      if (PR.CommissionDB && typeof PR.CommissionDB.findCategory === 'function') {
        result.categoryInfo = PR.CommissionDB.findCategory(PLATFORM, data.category);
      }
    } catch (err) {
      warn('Kategori arama hatasi:', err);
    }

    // Kar hesaplamasi
    try {
      if (PR.ProfitCalculator && typeof PR.ProfitCalculator.calculate === 'function') {
        result.profit = PR.ProfitCalculator.calculate({
          platform: PLATFORM,
          categoryKey: result.categoryInfo ? result.categoryInfo.key : '',
          categoryName: data.category,
          salePrice: data.price
        });
      }
    } catch (err) {
      warn('Kar hesaplama hatasi:', err);
    }

    // Satis tahmini
    try {
      if (PR.SalesEstimator && typeof PR.SalesEstimator.estimate === 'function') {
        result.sales = PR.SalesEstimator.estimate({
          platform: PLATFORM,
          categoryKey: result.categoryInfo ? result.categoryInfo.key : '',
          totalReviews: data.reviewCount,
          favoriteCount: data.favoriteCount || 0,
          currentPrice: data.price,
          averageRating: data.rating
        });
      }
    } catch (err) {
      warn('Satis tahmini hatasi:', err);
    }

    // Platform karsilastirmasi
    try {
      if (PR.ProfitCalculator && typeof PR.ProfitCalculator.compareAll === 'function') {
        result.comparison = PR.ProfitCalculator.compareAll({
          categoryName: data.category,
          categoryKey: result.categoryInfo ? result.categoryInfo.key : '',
          salePrice: data.price
        });
      }
    } catch (err) {
      warn('Platform karsilastirma hatasi:', err);
    }

    return result;
  }

  // ─── UI Enjeksiyonu ─────────────────────────────────────────

  /**
   * Arama sayfasina toplu analiz cubugu ve urun kartlarini enjekte eder.
   * @param {Object[]} products - Cikarilan urun listesi
   */
  function injectSearchUI(products) {
    try {
      if (!products || products.length === 0) return;
      if (!PR.InjectUI) {
        warn('InjectUI modulu yuklu degil');
        return;
      }

      // Onceki enjeksiyonlari temizle
      if (typeof PR.InjectUI.cleanup === 'function') {
        PR.InjectUI.cleanup();
      }

      // Analiz edilmis urunler
      var analyzedProducts = [];
      for (var i = 0; i < products.length; i++) {
        var analyzed = analyzeProduct(products[i]);
        if (analyzed) {
          analyzedProducts.push(analyzed);
        }
      }

      if (analyzedProducts.length === 0) return;

      // Toplam istatistikler
      var totalPrice = 0;
      var totalRating = 0;
      var ratedCount = 0;
      var totalReviews = 0;
      var avgProfit = 0;
      var profitCount = 0;

      for (var j = 0; j < analyzedProducts.length; j++) {
        var p = analyzedProducts[j];
        if (p.price > 0) totalPrice += p.price;
        if (p.rating > 0) {
          totalRating += p.rating;
          ratedCount++;
        }
        totalReviews += (p.reviewCount || 0);
        if (p.profit && p.profit.netMarginPercent != null) {
          avgProfit += p.profit.netMarginPercent;
          profitCount++;
        }
      }

      var aggregateData = {
        platform: PLATFORM,
        totalProducts: analyzedProducts.length,
        averagePrice: analyzedProducts.length > 0
          ? Math.round((totalPrice / analyzedProducts.length) * 100) / 100
          : 0,
        averageRating: ratedCount > 0
          ? Math.round((totalRating / ratedCount) * 10) / 10
          : 0,
        totalReviews: totalReviews,
        averageProfitMargin: profitCount > 0
          ? Math.round((avgProfit / profitCount) * 10) / 10
          : null,
        comparison: analyzedProducts[0] ? analyzedProducts[0].comparison : null
      };

      // Aggregate bar enjekte et
      if (typeof PR.InjectUI.createSearchAggregateBar === 'function') {
        PR.InjectUI.createSearchAggregateBar(aggregateData);
      }

      // Her urun kartina analiz karti enjekte et
      for (var k = 0; k < analyzedProducts.length; k++) {
        var product = analyzedProducts[k];
        var card = products[k].cardElement;

        if (!card || card.getAttribute(INJECTED_ATTR) === 'true') continue;

        card.setAttribute(INJECTED_ATTR, 'true');

        if (typeof PR.InjectUI.createSearchResultCard === 'function') {
          var uiCard = PR.InjectUI.createSearchResultCard(product);
          if (uiCard && card.parentElement) {
            card.parentElement.insertBefore(uiCard, card.nextSibling);
          }
        }
      }

      info('Arama UI enjekte edildi: ' + analyzedProducts.length + ' urun');
    } catch (err) {
      warn('Arama UI enjeksiyon hatasi:', err);
    }
  }

  // ─── Urun Skoru Hesaplama ─────────────────────────────────

  /**
   * Coklu faktorlu urun skoru hesaplar (0-10 arasi).
   * Faktorler: Rating (%25), Yorum hacmi (%20), Talep (%25), Trend (%15), Kar marji (%15)
   * @param {Object} params
   * @returns {number} 0-10 arasi skor
   */
  function calculateProductScore(params) {
    var scores = {};

    // 1. Rating skoru (%25 agirlik)
    var rating = Math.min(Math.max(params.rating || 0, 0), 5);
    if (rating === 0) {
      scores.rating = 5;
    } else if (rating >= 4.5) {
      scores.rating = 10;
    } else if (rating >= 4.0) {
      scores.rating = 8.5;
    } else if (rating >= 3.5) {
      scores.rating = 7;
    } else if (rating >= 3.0) {
      scores.rating = 5.5;
    } else if (rating >= 2.0) {
      scores.rating = 3.5;
    } else {
      scores.rating = 2;
    }

    // 2. Yorum hacmi skoru (%20 agirlik)
    var reviews = params.reviewCount || 0;
    if (reviews >= 1000) {
      scores.reviews = 10;
    } else if (reviews >= 500) {
      scores.reviews = 9;
    } else if (reviews >= 200) {
      scores.reviews = 8;
    } else if (reviews >= 100) {
      scores.reviews = 7;
    } else if (reviews >= 50) {
      scores.reviews = 6;
    } else if (reviews >= 20) {
      scores.reviews = 5;
    } else if (reviews >= 5) {
      scores.reviews = 3.5;
    } else {
      scores.reviews = 2;
    }

    // 3. Talep seviyesi skoru (%25 agirlik)
    var demandScores = {
      'very_high': 10, 'high': 8, 'medium': 6, 'low': 4, 'very_low': 2
    };
    scores.demand = demandScores[params.demandLevel] || 5;

    // 4. Trend skoru (%15 agirlik)
    var trendScores = { 'up': 9, 'stable': 6, 'down': 3 };
    scores.trend = trendScores[params.trend] || 6;

    // 5. Fiyat/Deger skoru (%15 agirlik)
    var commissionRate = params.commissionRate || 0;
    if (commissionRate <= 5) {
      scores.value = 9;
    } else if (commissionRate <= 10) {
      scores.value = 7.5;
    } else if (commissionRate <= 15) {
      scores.value = 6;
    } else if (commissionRate <= 25) {
      scores.value = 4.5;
    } else {
      scores.value = 3;
    }
    if (params.originalPrice > 0 && params.price > 0 && params.originalPrice > params.price) {
      var discountPct = ((params.originalPrice - params.price) / params.originalPrice) * 100;
      if (discountPct >= 10) {
        scores.value = Math.min(10, scores.value + 1);
      }
    }

    var weightedScore = (
      scores.rating * 0.25 +
      scores.reviews * 0.20 +
      scores.demand * 0.25 +
      scores.trend * 0.15 +
      scores.value * 0.15
    );

    // 0-5 araliginda sinirla (platformlarla uyumlu), 1 ondalik basamak
    var finalScore = Math.max(0, Math.min(10, weightedScore)) / 2;
    return Math.round(finalScore * 10) / 10;
  }

  /**
   * Ic analiz nesnesini createProductPanel'in beklediği duz formata donusturur.
   * @param {Object} analysis - analyzeProduct sonucu
   * @returns {Object} createProductPanel icin duz veri nesnesi
   */
  function flattenAnalysisForPanel(analysis) {
    var profit = analysis.profit || {};
    var sales = analysis.sales || {};

    // Coklu faktorlu urun skoru hesaplama (0-10)
    var productScore = calculateProductScore({
      rating: analysis.rating || 0,
      reviewCount: analysis.reviewCount || 0,
      favoriteCount: analysis.favoriteCount || 0,
      demandLevel: sales.demandLevel || 'very_low',
      trend: sales.trend || 'stable',
      estimatedMonthlySales: sales.estimatedMonthlySales || 0,
      price: analysis.price || 0,
      originalPrice: analysis.originalPrice || 0,
      commissionRate: profit.commissionRate || 0
    });

    return {
      productScore: productScore,
      estimatedMonthlySales: sales.estimatedMonthlySales || 0,
      estimatedMonthlyRevenue: sales.estimatedMonthlyRevenue || 0,
      reviewRate: analysis.reviewCount || 0,
      favoriteRate: analysis.favoriteCount || 0,
      trend: sales.trend || 'stable',
      salePrice: profit.salePrice || analysis.price || 0,
      commissionRate: profit.commissionRate || 0,
      commissionAmount: profit.commissionAmount || 0,
      commissionKDV: profit.commissionKDV || 0,
      platformFee: profit.platformFee || 0,
      platformFeeKDV: profit.platformFeeKDV || 0,
      cargoFee: profit.estimatedCargo || 0,
      netRevenue: profit.netRevenue || 0,
      comparison: analysis.comparison || null,
      priceHistory: null,
      isPro: userIsPro,
      trialDaysLeft: userTrialDaysLeft
    };
  }

  /**
   * Urun detay sayfasina analiz paneli enjekte eder.
   * @param {Object} productData - Cikarilan urun detay verisi
   */
  function injectProductUI(productData) {
    try {
      if (!productData) return;
      if (!PR.InjectUI) {
        warn('InjectUI modulu yuklu degil');
        return;
      }

      // Onceki enjeksiyonlari temizle
      if (typeof PR.InjectUI.cleanup === 'function') {
        PR.InjectUI.cleanup();
      }

      var analysis = analyzeProduct(productData);
      if (!analysis) return;

      // Ana urun paneli - duz formata cevir ve DOM'a ekle
      if (typeof PR.InjectUI.createProductPanel === 'function') {
        var panelData = flattenAnalysisForPanel(analysis);

        // Limit kontrolu - background'a sor, sonra paneli render et
        if (typeof chrome !== 'undefined' && chrome.runtime && chrome.runtime.sendMessage) {
          chrome.runtime.sendMessage({
            type: 'peek_usage',
            payload: { usageType: 'productAnalyses' }
          }, function (response) {
            if (chrome.runtime.lastError || !response) {
              renderN11Panel(panelData);
              return;
            }
            if (!response.allowed) {
              panelData.limitReached = true;
              panelData.limitInfo = response;
            }
            renderN11Panel(panelData);
          });
        } else {
          renderN11Panel(panelData);
        }
      }

      info('Urun detay UI enjekte edildi: ' + (analysis.title || ''));
    } catch (err) {
      warn('Urun detay UI enjeksiyon hatasi:', err);
    }
  }

  /**
   * N11 panel DOM'a ekleme yardimcisi.
   * @param {Object} panelData - Panel verisi
   */
  function renderN11Panel(panelData) {
    try {
      var panelEl = PR.InjectUI.createProductPanel(panelData);
      if (panelEl) {
        var target = document.querySelector('.proDetail')
          || document.querySelector('#product-detail')
          || document.querySelector('main')
          || document.body;
        target.appendChild(panelEl);
      }
    } catch (err) {
      warn('N11 panel render hatasi:', err);
    }
  }

  /**
   * Magaza sayfasina magaza analiz paneli enjekte eder.
   * @param {Object} storeData - Cikarilan magaza verisi
   */
  function injectStoreUI(storeData) {
    try {
      if (!storeData) return;
      if (!PR.InjectUI) {
        warn('InjectUI modulu yuklu degil');
        return;
      }

      if (typeof PR.InjectUI.cleanup === 'function') {
        PR.InjectUI.cleanup();
      }

      // Magaza urunlerini analiz et
      var analyzedProducts = [];
      for (var i = 0; i < storeData.products.length; i++) {
        var analyzed = analyzeProduct(storeData.products[i]);
        if (analyzed) {
          analyzedProducts.push(analyzed);
        }
      }

      var storeAnalysis = {
        platform: PLATFORM,
        storeName: storeData.storeName,
        storeUrl: storeData.storeUrl,
        productCount: storeData.productCount,
        products: analyzedProducts
      };

      if (typeof PR.InjectUI.createStorePanel === 'function') {
        PR.InjectUI.createStorePanel(storeAnalysis);
      }

      info('Magaza UI enjekte edildi: ' + storeData.storeName);
    } catch (err) {
      warn('Magaza UI enjeksiyon hatasi:', err);
    }
  }

  // ─── Background Script Iletisimi ───────────────────────────

  /**
   * Cikarilan veriyi background script'e iletir.
   * @param {string} type - Mesaj tipi ('search' | 'product' | 'store')
   * @param {Object} data - Gonderilecek veri
   */
  function sendToBackground(type, data) {
    try {
      if (typeof chrome === 'undefined' || !chrome.runtime || !chrome.runtime.sendMessage) {
        return;
      }

      // cardElement gibi DOM referanslarini temizle (JSON serialize edilemez)
      var cleanData = JSON.parse(JSON.stringify(data, function (_key, value) {
        if (value instanceof HTMLElement) return undefined;
        return value;
      }));

      chrome.runtime.sendMessage({
        type: 'product_data',
        platform: PLATFORM,
        pageType: type,
        data: cleanData,
        url: location.href,
        timestamp: Date.now()
      }, function (_response) {
        // Callback hatalarini yut - background yanit vermese bile sorun degil
        if (chrome.runtime.lastError) {
          // Sessizce gec
        }
      });
    } catch (err) {
      warn('Background iletisim hatasi:', err);
    }
  }

  // ─── Sayfa Yoneticileri ─────────────────────────────────────

  /**
   * Arama sayfasi isleyicisi.
   */
  function handleSearch() {
    info('Arama sayfasi tespit edildi');

    var products = extractSearchProducts();
    if (products.length === 0) {
      // N11 bazen icerigi gecikmeyle yukler; yeniden dene
      setTimeout(function () {
        var retryProducts = extractSearchProducts();
        if (retryProducts.length > 0) {
          injectSearchUI(retryProducts);
          sendToBackground(PAGE_TYPES.SEARCH, retryProducts);
        }
      }, 1500);
      return;
    }

    injectSearchUI(products);
    sendToBackground(PAGE_TYPES.SEARCH, products);
  }

  /**
   * Urun detay sayfasi isleyicisi.
   */
  function handleProduct() {
    info('Urun detay sayfasi tespit edildi');

    var productData = extractProductDetail();
    if (productData) {
      injectProductUI(productData);
      sendToBackground(PAGE_TYPES.PRODUCT, productData);
      tryTrackProduct(productData);
      return;
    }

    // Vue.js render gecikmesi icin kademeli retry - ilk deneme hizli
    var retryDelays = [200, 600, 1500, 3500];
    var retryIndex = 0;

    function retryExtract() {
      if (retryIndex >= retryDelays.length) {
        warn('Urun verisi ' + retryDelays.length + ' denemede alinamadi');
        return;
      }
      var delay = retryDelays[retryIndex];
      retryIndex++;
      setTimeout(function () {
        var retryData = extractProductDetail();
        if (retryData) {
          info('Urun verisi ' + retryIndex + '. denemede alindi');
          injectProductUI(retryData);
          sendToBackground(PAGE_TYPES.PRODUCT, retryData);
          tryTrackProduct(retryData);
        } else {
          retryExtract();
        }
      }, delay);
    }

    retryExtract();
  }

  /**
   * Magaza sayfasi isleyicisi.
   */
  function handleStore() {
    info('Magaza sayfasi tespit edildi');

    var storeData = extractStoreData();
    if (!storeData) {
      setTimeout(function () {
        var retryData = extractStoreData();
        if (retryData) {
          injectStoreUI(retryData);
          sendToBackground(PAGE_TYPES.STORE, retryData);
        }
      }, 1500);
      return;
    }

    injectStoreUI(storeData);
    sendToBackground(PAGE_TYPES.STORE, storeData);
  }

  /**
   * Urunu stok takibine almaya calisir.
   * @param {Object} productData
   */
  function tryTrackProduct(productData) {
    try {
      if (!productData || !productData.productId) return;
      if (!PR.SalesEstimator || typeof PR.SalesEstimator.trackProduct !== 'function') return;

      PR.SalesEstimator.trackProduct(
        productData.productId,
        PLATFORM,
        0 // N11'de stok bilgisi genelde acik degildir
      );
    } catch (err) {
      warn('Stok takip hatasi:', err);
    }
  }

  // ─── SPA Navigasyon Yonetimi ───────────────────────────────

  /**
   * N11 geleneksel sayfa yuklemeleri kullanir ancak
   * dinamik icerik yukleme ihtimaline karsi MutationObserver
   * guvenlik agi olarak kullanilir.
   */

  /** Mevcut URL takibi */
  var currentUrl = location.href;

  /** Sayfa iceriginin degisimini izleyen observer */
  var observer = new MutationObserver(debounce(function () {
    try {
      if (location.href !== currentUrl) {
        currentUrl = location.href;
        info('Navigasyon degisimi tespit edildi: ' + currentUrl);

        // Onceki enjeksiyonlari temizle
        if (PR.InjectUI && typeof PR.InjectUI.cleanup === 'function') {
          PR.InjectUI.cleanup();
        }

        // Yeni icerigin yuklenmesini bekle
        setTimeout(init, SPA_REINJECT_DELAY);
      }
    } catch (err) {
      warn('Navigation observer hatasi:', err);
    }
  }, MUTATION_DEBOUNCE_MS));

  /**
   * N11 sayfalama (pagination) ve lazy-load icin ek observer.
   * Yeni urun kartlari yuklendikce bunlari da analiz eder.
   */
  var contentObserver = new MutationObserver(debounce(function () {
    try {
      // Sayfa URL'si degismediyse ama yeni kartlar eklendiyse
      if (location.href === currentUrl && detectPageType() === PAGE_TYPES.SEARCH) {
        var uninjectCards = document.querySelectorAll(
          SELECTORS.productCard[0] + ':not([' + INJECTED_ATTR + '])'
        );

        // Fallback seciciler dene
        if (!uninjectCards || uninjectCards.length === 0) {
          for (var s = 1; s < SELECTORS.productCard.length; s++) {
            try {
              uninjectCards = document.querySelectorAll(
                SELECTORS.productCard[s] + ':not([' + INJECTED_ATTR + '])'
              );
              if (uninjectCards && uninjectCards.length > 0) break;
            } catch (_e) {
              // Gecersiz secici
            }
          }
        }

        if (uninjectCards && uninjectCards.length > 0) {
          info('Yeni urun kartlari tespit edildi: ' + uninjectCards.length);

          var newProducts = [];
          for (var i = 0; i < uninjectCards.length; i++) {
            var product = extractProductFromCard(uninjectCards[i]);
            if (product) {
              newProducts.push(product);
            }
          }

          if (newProducts.length > 0) {
            // Sadece yeni kartlara enjeksiyon yap
            for (var j = 0; j < newProducts.length; j++) {
              var card = newProducts[j].cardElement;
              if (!card || card.getAttribute(INJECTED_ATTR) === 'true') continue;

              card.setAttribute(INJECTED_ATTR, 'true');

              var analyzed = analyzeProduct(newProducts[j]);
              if (analyzed && PR.InjectUI && typeof PR.InjectUI.createSearchResultCard === 'function') {
                var uiCard = PR.InjectUI.createSearchResultCard(analyzed);
                if (uiCard && card.parentElement) {
                  card.parentElement.insertBefore(uiCard, card.nextSibling);
                }
              }
            }
          }
        }
      }
    } catch (err) {
      warn('Icerik observer hatasi:', err);
    }
  }, 600));

  // ─── Baslatma ───────────────────────────────────────────────

  /**
   * Ana baslatma fonksiyonu. Sayfa tipini tespit edip ilgili isleyiciyi calistirir.
   */
  function init() {
    try {
      var pageType = detectPageType();
      info('Sayfa tipi: ' + pageType);

      switch (pageType) {
        case PAGE_TYPES.SEARCH:
          handleSearch();
          break;
        case PAGE_TYPES.PRODUCT:
          handleProduct();
          break;
        case PAGE_TYPES.STORE:
          handleStore();
          break;
        default:
          info('Desteklenmeyen sayfa tipi, islem yapilmiyor');
          break;
      }
    } catch (err) {
      warn('Baslatma hatasi:', err);
    }
  }

  // DOM hazir oldugundan emin ol
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  // Navigasyon degisikliklerini izle
  try {
    if (document.body) {
      observer.observe(document.body, { childList: true, subtree: true });
      contentObserver.observe(document.body, { childList: true, subtree: true });
    } else {
      document.addEventListener('DOMContentLoaded', function () {
        observer.observe(document.body, { childList: true, subtree: true });
        contentObserver.observe(document.body, { childList: true, subtree: true });
      });
    }
  } catch (err) {
    warn('MutationObserver baslatilamadi:', err);
  }

  info('N11 content script yuklendi');

})();
