/**
 * PazarRadar - Hepsiburada Content Script
 *
 * Hepsiburada uzerinde calisan icerik betigi. Urun arama sonuclari,
 * urun detay sayfasi ve magaza sayfalari uzerinden veri cikarip
 * PazarRadar analiz panellerini sayfaya enjekte eder.
 *
 * Bagimliliklar:
 *   - window.PazarRadar.CommissionDB
 *   - window.PazarRadar.ProfitCalculator
 *   - window.PazarRadar.SalesEstimator
 *   - window.PazarRadar.InjectUI
 *
 * @namespace HepsiburadaContentScript
 */
(function () {
  'use strict';

  // ─── Global Erisim & Guvensizlik Korumalari ────────────────
  var PR;
  try {
    PR = window.PazarRadar;
  } catch (_err) {
    return;
  }

  if (!PR) {
    return;
  }

  var PLATFORM = 'hepsiburada';
  var LOG_PREFIX = '[PazarRadar][HB]';

  /** Production modunda konsol ciktisini devre disi birakir */
  var DEBUG = false;

  /** Lisans durumu - chrome.storage'dan yuklenir */
  var userIsPro = false;
  var userTrialDaysLeft = 0;

  try {
    if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.local) {
      chrome.storage.local.get('license', function (result) {
        var license = result && result.license;
        if (license && license.plan && license.plan !== 'free') {
          if (!license.validUntil || Date.now() <= license.validUntil) {
            userIsPro = true;
          }
          // Trial gun hesabi
          if (license.plan === 'trial' && license.trialEndsAt) {
            var msLeft = license.trialEndsAt - Date.now();
            userTrialDaysLeft = Math.max(0, Math.ceil(msLeft / (24 * 60 * 60 * 1000)));
            if (userTrialDaysLeft > 0) { userIsPro = true; }
          }
        }
      });
    }
  } catch (_e) { /* noop */ }

  // ─── Sabitler ───────────────────────────────────────────────

  /** Sayfa tipi sabitleri */
  var PAGE_TYPES = Object.freeze({
    SEARCH: 'search',
    PRODUCT: 'product',
    STORE: 'store',
    UNKNOWN: 'unknown'
  });

  /**
   * CSS secici havuzu. Hepsiburada'nin farkli versiyonlarinda calismasi icin
   * birden fazla secici tanimlanir; ilk eslesen kullanilir.
   */
  var SELECTORS = Object.freeze({
    // Arama sonuclari
    searchResultContainer: [
      '[data-test-id="product-listing"]',
      '.searchResultList',
      '.product-list',
      '#pagination-wrapper',
      '.search-result-wrapper'
    ],
    productCard: [
      '[data-test-id="product-card"]',
      '.product-card',
      'li.search-item',
      '.productListContent-item'
    ],
    productTitle: [
      '[data-test-id="product-card-name"]',
      '.product-title',
      '.product-card-name',
      'h3.product-title',
      '.hb-pl-card-title'
    ],
    productPrice: [
      '[data-test-id="price-current-price"]',
      '.product-price',
      '.price-value',
      '.hb-pl-card-price'
    ],
    productOriginalPrice: [
      '[data-test-id="price-old-price"]',
      '.product-old-price',
      '.price-old',
      '.hb-pl-card-oldprice'
    ],
    productLink: [
      'a[href*="/p-"]',
      'a.product-card-link',
      'a[data-test-id="product-card-link"]'
    ],
    productRating: [
      '[data-test-id="product-card-rating"]',
      '.ratings',
      '.product-rating',
      '.star-w'
    ],
    productReviewCount: [
      '[data-test-id="product-card-review-count"]',
      '.review-count',
      '.ratings-count'
    ],

    // Urun detay sayfasi
    detailTitle: [
      'h1[data-test-id="title"]',
      'h1[data-test-id="product-name"]',
      'h1.product-name',
      'h1#product-name',
      '.product-detail-name h1',
      'span[data-bind="markupText: \'productName\'"]'
    ],
    detailPrice: [
      '[data-test-id="price"]',
      '[data-test-id="price-current-price"]',
      '.product-price',
      '#offering-price .price-value',
      'span[data-bind="markupText:\'currentPriceBeforePoint\'"]'
    ],
    detailOriginalPrice: [
      '[data-test-id="price-old-price"]',
      '.product-old-price',
      '.price-old-value'
    ],
    detailRating: [
      '[data-test-id="has-review"] > :first-child',
      '[data-test-id="has-review"]',
      '[data-test-id="product-rating"]',
      '[class*="hermes-AverageRateBox"]',
      '.hermes-RatingBar-module',
      '.rating-line .star-w',
      '#productReviews .star-w'
    ],
    detailReviewCount: [
      'a[href*="-yorumlari"]',
      '[data-test-id="review-count"]',
      '.hermes-ReviewSummary-module',
      '#productReviews .total-review-count',
      '.review-count'
    ],
    detailSeller: [
      '[data-test-id="buyBox-seller"] a[href*="/magaza/"]',
      '.merchant-name',
      '[data-test-id="merchant-name"]',
      '.seller-name-text',
      '.merchant-box .merchant-name-text'
    ],
    detailBreadcrumb: [
      '[data-test-id="breadcrumb"]',
      'main ul:has(> li > a[href="/"])',
      'main ul',
      '.breadcrumb',
      '.breadcrumb-wrapper',
      'nav[aria-label="breadcrumb"]'
    ],
    detailSpecTable: [
      '.data-list',
      '.technical-spec',
      '[data-test-id="product-properties"]',
      '.product-feature-group'
    ],

    // Diger saticilar (Buy Box)
    otherSellersSection: [
      '[data-test-id="other-sellers"]',
      '.other-sellers',
      '.merchant-list',
      '.other-sellers-container'
    ],
    otherSellerItem: [
      '[data-test-id="other-seller-item"]',
      '.other-seller-item',
      '.merchant-list-item'
    ],
    otherSellerName: [
      '[data-test-id="other-seller-name"]',
      '.merchant-name',
      '.seller-name'
    ],
    otherSellerPrice: [
      '[data-test-id="other-seller-price"]',
      '.other-seller-price',
      '.seller-price'
    ],

    // Magaza sayfasi
    storeName: [
      '.store-name',
      '.merchant-detail-name',
      '[data-test-id="store-name"]',
      'h1.store-title'
    ],
    storeProducts: [
      '.product-card',
      '[data-test-id="product-card"]',
      '.store-product-item'
    ]
  });

  /** Hepsiburada URL desenleri */
  var URL_PATTERNS = Object.freeze({
    search: /hepsiburada\.com\/(ara\?|kategori-|.*\?.*q=)/i,
    product: /hepsiburada\.com\/.*-p-[A-Z0-9]+/i,
    store: /hepsiburada\.com\/magaza\//i
  });

  /** Urun ID regex deseni - HBCV... formati */
  var PRODUCT_ID_REGEX = /-p-([A-Z0-9]+)/i;

  /** Enjeksiyon isareti - birden fazla calismayi onle */
  var INJECTED_ATTR = 'data-pazar-radar-injected';

  /** SPA gecislerinde yeniden enjeksiyon icin bekleme suresi (ms) */
  var SPA_REINJECT_DELAY = 800;

  /** MutationObserver debounce suresi (ms) */
  var MUTATION_DEBOUNCE_MS = 400;

  // ─── Yardimci Fonksiyonlar ──────────────────────────────────

  /**
   * Birden fazla secici arasinda ilk eslesen elementi bulur.
   * @param {Element|Document} context - Arama yapilacak ust element
   * @param {string[]} selectorList - Denenecek CSS seciciler
   * @returns {Element|null}
   */
  function queryFirst(context, selectorList) {
    if (!context || !selectorList) return null;

    for (var i = 0; i < selectorList.length; i++) {
      try {
        var el = context.querySelector(selectorList[i]);
        if (el) return el;
      } catch (_e) {
        // Gecersiz secici - bir sonrakini dene
      }
    }
    return null;
  }

  /**
   * Birden fazla secici arasinda ilk eslesen tum elementleri bulur.
   * @param {Element|Document} context - Arama yapilacak ust element
   * @param {string[]} selectorList - Denenecek CSS seciciler
   * @returns {Element[]}
   */
  function queryAllFirst(context, selectorList) {
    if (!context || !selectorList) return [];

    for (var i = 0; i < selectorList.length; i++) {
      try {
        var els = context.querySelectorAll(selectorList[i]);
        if (els && els.length > 0) return Array.from(els);
      } catch (_e) {
        // Gecersiz secici - bir sonrakini dene
      }
    }
    return [];
  }

  /**
   * Element metin icerigini temizleyerek dondurur.
   * @param {Element|null} el
   * @returns {string}
   */
  function getTextContent(el) {
    if (!el) return '';
    return (el.textContent || '').replace(/\s+/g, ' ').trim();
  }

  /**
   * Fiyat metninden sayisal deger cikarir.
   * "1.299,90 TL" => 1299.90
   * @param {string} priceText
   * @returns {number}
   */
  function parsePrice(priceText) {
    if (!priceText) return 0;

    var cleaned = priceText
      .replace(/[^\d.,]/g, '')  // Sadece rakam, nokta, virgul birak
      .replace(/\./g, '')       // Binlik ayiraci kaldir
      .replace(',', '.');       // Ondalik virgulu noktaya cevir

    var value = parseFloat(cleaned);
    return Number.isFinite(value) && value > 0 ? value : 0;
  }

  /**
   * Yildiz puanini cikarir. "4.5" veya CSS width'ten hesaplar.
   * @param {Element|null} ratingEl
   * @returns {number}
   */
  function parseRating(ratingEl) {
    if (!ratingEl) return 0;

    // Metin iceriginden dene
    var text = getTextContent(ratingEl);
    var match = text.match(/([\d]+[.,]?\d*)/);
    if (match) {
      var val = parseFloat(match[1].replace(',', '.'));
      if (Number.isFinite(val) && val >= 0 && val <= 5) return val;
    }

    // aria-label'dan dene
    var ariaLabel = ratingEl.getAttribute('aria-label') || '';
    var ariaMatch = ariaLabel.match(/([\d]+[.,]?\d*)/);
    if (ariaMatch) {
      var ariaVal = parseFloat(ariaMatch[1].replace(',', '.'));
      if (Number.isFinite(ariaVal) && ariaVal >= 0 && ariaVal <= 5) return ariaVal;
    }

    // Star width oranina bak (bazi sitelerde %80 = 4 yildiz)
    var starFill = ratingEl.querySelector('.star-fill, .full, [style*="width"]');
    if (starFill) {
      var style = starFill.getAttribute('style') || '';
      var widthMatch = style.match(/width:\s*([\d.]+)%/);
      if (widthMatch) {
        return parseFloat(widthMatch[1]) / 20; // %100 = 5 yildiz
      }
    }

    return 0;
  }

  /**
   * Yorum/degerlendirme sayisini cikarir.
   * "(245)" veya "245 degerlendirme" => 245
   * @param {Element|null} countEl
   * @returns {number}
   */
  function parseReviewCount(countEl) {
    if (!countEl) return 0;

    var text = getTextContent(countEl);
    var match = text.match(/([\d.]+)/);
    if (match) {
      var val = parseInt(match[1].replace(/\./g, ''), 10);
      return Number.isFinite(val) ? val : 0;
    }
    return 0;
  }

  /**
   * Breadcrumb'dan kategori bilgisini cikarir.
   * @param {Element|Document} context
   * @returns {string}
   */
  function extractCategoryFromBreadcrumb(context) {
    var breadcrumb = queryFirst(context, SELECTORS.detailBreadcrumb);
    if (!breadcrumb) return '';

    var items = breadcrumb.querySelectorAll('a, span, li');
    if (!items || items.length === 0) return '';

    // Son anlamli breadcrumb ogesi kategoridir (ilk oge genelde "Anasayfa")
    var categories = [];
    for (var i = 0; i < items.length; i++) {
      var text = getTextContent(items[i]);
      if (text && text.toLowerCase() !== 'anasayfa' && text !== '>') {
        categories.push(text);
      }
    }

    // Son kategoriyi dondur (en spesifik)
    return categories.length > 0 ? categories[categories.length - 1] : '';
  }

  /**
   * URL'den veya data attribute'den urun ID'sini cikarir.
   * @param {string} url
   * @param {Element} [card] - Urun karti elementi
   * @returns {string}
   */
  function extractProductId(url, card) {
    // data-productid attribute'undan dene
    if (card) {
      var dataId = card.getAttribute('data-productid') ||
                   card.getAttribute('data-product-id') ||
                   card.getAttribute('data-sku');
      if (dataId) return dataId;
    }

    // URL'den cikar: .../urun-adi-p-HBCV00000XXXXX
    if (url) {
      var match = url.match(PRODUCT_ID_REGEX);
      if (match) return match[1];
    }

    return '';
  }

  /**
   * JSON-LD yapisal verisini sayfadan cikarir.
   * @returns {Object|null}
   */
  function extractJsonLd() {
    try {
      var scripts = document.querySelectorAll('script[type="application/ld+json"]');
      for (var i = 0; i < scripts.length; i++) {
        var data = JSON.parse(scripts[i].textContent || '{}');

        // Dizi icerisinde Product araniyor
        if (Array.isArray(data)) {
          for (var j = 0; j < data.length; j++) {
            if (data[j]['@type'] === 'Product') return data[j];
          }
        }

        // Direkt Product nesnesi
        if (data['@type'] === 'Product') return data;

        // @graph icinde arama
        if (data['@graph'] && Array.isArray(data['@graph'])) {
          for (var k = 0; k < data['@graph'].length; k++) {
            if (data['@graph'][k]['@type'] === 'Product') return data['@graph'][k];
          }
        }
      }
    } catch (_err) {
      warn('JSON-LD parse hatasi');
    }
    return null;
  }

  /**
   * Konsola uyari logu yazar.
   * @param {...*} args
   */
  function warn() {
    if (!DEBUG) return;
    var args = [LOG_PREFIX].concat(Array.from(arguments));
    try { console.warn.apply(console, args); } catch (_e) { /* noop */ }
  }

  /**
   * Konsola bilgi logu yazar.
   * @param {...*} args
   */
  function info() {
    if (!DEBUG) return;
    var args = [LOG_PREFIX].concat(Array.from(arguments));
    try { console.info.apply(console, args); } catch (_e) { /* noop */ }
  }

  /**
   * Debounce fonksiyonu.
   * @param {Function} fn
   * @param {number} ms
   * @returns {Function}
   */
  function debounce(fn, ms) {
    var timer;
    return function () {
      var args = arguments;
      var self = this;
      clearTimeout(timer);
      timer = setTimeout(function () {
        fn.apply(self, args);
      }, ms);
    };
  }

  // ─── Sayfa Tipi Tespiti ─────────────────────────────────────

  /**
   * Mevcut URL'ye gore sayfa tipini tespit eder.
   * @returns {string} PAGE_TYPES degerlerinden biri
   */
  function detectPageType() {
    var url = location.href;

    if (URL_PATTERNS.product.test(url)) return PAGE_TYPES.PRODUCT;
    if (URL_PATTERNS.store.test(url)) return PAGE_TYPES.STORE;
    if (URL_PATTERNS.search.test(url)) return PAGE_TYPES.SEARCH;

    // URL deseniyle eslesmeyen ama urun listesi iceren sayfa olabilir
    var container = queryFirst(document, SELECTORS.searchResultContainer);
    if (container) return PAGE_TYPES.SEARCH;

    return PAGE_TYPES.UNKNOWN;
  }

  // ─── Veri Cikarma: Arama Sonuclari ─────────────────────────

  /**
   * Tek bir urun kartindan veri cikarir.
   * @param {Element} card - Urun karti DOM elementi
   * @returns {Object|null} Cikarilan urun verisi veya null
   */
  function extractProductFromCard(card) {
    try {
      if (!card) return null;

      var titleEl = queryFirst(card, SELECTORS.productTitle);
      var priceEl = queryFirst(card, SELECTORS.productPrice);
      var originalPriceEl = queryFirst(card, SELECTORS.productOriginalPrice);
      var linkEl = queryFirst(card, SELECTORS.productLink);
      var ratingEl = queryFirst(card, SELECTORS.productRating);
      var reviewCountEl = queryFirst(card, SELECTORS.productReviewCount);

      var title = getTextContent(titleEl);
      var price = parsePrice(getTextContent(priceEl));

      // Baslik veya fiyat yoksa gecersiz kart
      if (!title && price <= 0) return null;

      var url = (linkEl && linkEl.href) ? linkEl.href : '';
      var productId = extractProductId(url, card);

      return {
        platform: PLATFORM,
        title: title,
        price: price,
        originalPrice: parsePrice(getTextContent(originalPriceEl)),
        rating: parseRating(ratingEl),
        reviewCount: parseReviewCount(reviewCountEl),
        seller: '',
        url: url,
        productId: productId,
        category: extractCategoryFromBreadcrumb(document),
        favoriteCount: 0,
        cardElement: card
      };
    } catch (err) {
      warn('Kart verisi cikarma hatasi:', err);
      return null;
    }
  }

  /**
   * Arama sayfasindaki tum urun kartlarindan veri cikarir.
   * @returns {Object[]} Cikarilan urun listesi
   */
  function extractSearchProducts() {
    var cards = queryAllFirst(document, SELECTORS.productCard);

    if (cards.length === 0) {
      warn('Urun karti bulunamadi');
      return [];
    }

    var products = [];
    for (var i = 0; i < cards.length; i++) {
      var product = extractProductFromCard(cards[i]);
      if (product) {
        products.push(product);
      }
    }

    info('Toplam ' + products.length + ' urun cikarildi');
    return products;
  }

  // ─── Veri Cikarma: Urun Detay ──────────────────────────────

  /**
   * "Diger Saticilar" bolumunden tum satici ve fiyat bilgilerini cikarir.
   * Bu Hepsiburada'ya ozgu Buy Box analizidir.
   * @returns {Object[]} Satici listesi
   */
  function extractOtherSellers() {
    var sellers = [];

    try {
      var section = queryFirst(document, SELECTORS.otherSellersSection);
      if (!section) return sellers;

      var items = queryAllFirst(section, SELECTORS.otherSellerItem);

      for (var i = 0; i < items.length; i++) {
        var nameEl = queryFirst(items[i], SELECTORS.otherSellerName);
        var priceEl = queryFirst(items[i], SELECTORS.otherSellerPrice);

        var name = getTextContent(nameEl);
        var price = parsePrice(getTextContent(priceEl));

        if (name || price > 0) {
          sellers.push({
            name: name || 'Bilinmeyen Satici',
            price: price,
            index: i
          });
        }
      }
    } catch (err) {
      warn('Diger saticilar cikarma hatasi:', err);
    }

    return sellers;
  }

  /**
   * Urun ozellikler/spesifikasyon tablosunu cikarir.
   * @returns {Object} Anahtar-deger ciftleri
   */
  function extractSpecifications() {
    var specs = {};

    try {
      var specTable = queryFirst(document, SELECTORS.detailSpecTable);
      if (!specTable) return specs;

      // Tablo satirlari veya dt/dd ciftleri uzerinde gezin
      var rows = specTable.querySelectorAll('tr, .data-list-item, li');
      for (var i = 0; i < rows.length; i++) {
        var cells = rows[i].querySelectorAll('td, th, span, .attr-name, .attr-value');
        if (cells.length >= 2) {
          var key = getTextContent(cells[0]);
          var value = getTextContent(cells[1]);
          if (key) {
            specs[key] = value;
          }
        }
      }

      // dt/dd yapisi desteyi
      var dtElements = specTable.querySelectorAll('dt');
      var ddElements = specTable.querySelectorAll('dd');
      var len = Math.min(dtElements.length, ddElements.length);
      for (var j = 0; j < len; j++) {
        var dtText = getTextContent(dtElements[j]);
        var ddText = getTextContent(ddElements[j]);
        if (dtText) {
          specs[dtText] = ddText;
        }
      }
    } catch (err) {
      warn('Spesifikasyon cikarma hatasi:', err);
    }

    return specs;
  }

  /**
   * Urun detay sayfasindan tum verileri cikarir.
   * JSON-LD, DOM elementleri ve Buy Box bilgisini birlestirir.
   * @returns {Object|null} Urun detay verisi
   */
  function extractProductDetail() {
    try {
      // Once JSON-LD'den zengin veri almaya calis
      var jsonLd = extractJsonLd();

      // DOM'dan cikarma
      var titleEl = queryFirst(document, SELECTORS.detailTitle);
      var priceEl = queryFirst(document, SELECTORS.detailPrice);
      var originalPriceEl = queryFirst(document, SELECTORS.detailOriginalPrice);
      var ratingEl = queryFirst(document, SELECTORS.detailRating);
      var reviewCountEl = queryFirst(document, SELECTORS.detailReviewCount);
      var sellerEl = queryFirst(document, SELECTORS.detailSeller);

      var title = getTextContent(titleEl);
      var price = parsePrice(getTextContent(priceEl));

      // JSON-LD varsa DOM verisini zenginlestir
      if (jsonLd) {
        if (!title && jsonLd.name) {
          title = jsonLd.name;
        }
        if (price <= 0 && jsonLd.offers) {
          var offers = Array.isArray(jsonLd.offers) ? jsonLd.offers[0] : jsonLd.offers;
          if (offers && offers.price) {
            price = parseFloat(offers.price) || 0;
          }
        }
      }

      if (!title && price <= 0) {
        warn('Urun detayi cikarilacak veri bulunamadi');
        return null;
      }

      // Fiyat bulunamadiysa (React/SPA gecikme) null don ki retry tetiklensin
      if (price <= 0) {
        warn('Fiyat henuz yuklenemedi, retry beklenecek');
        return null;
      }

      var rating = parseRating(ratingEl);
      var reviewCount = parseReviewCount(reviewCountEl);

      // JSON-LD'den rating/review zenginlestirmesi
      if (jsonLd && jsonLd.aggregateRating) {
        if (rating <= 0 && jsonLd.aggregateRating.ratingValue) {
          rating = parseFloat(jsonLd.aggregateRating.ratingValue) || 0;
        }
        if (reviewCount <= 0 && jsonLd.aggregateRating.reviewCount) {
          reviewCount = parseInt(jsonLd.aggregateRating.reviewCount, 10) || 0;
        }
      }

      var productId = extractProductId(location.href, null);
      var category = extractCategoryFromBreadcrumb(document);
      var seller = getTextContent(sellerEl);

      // Hepsiburada'ya ozgu: Buy Box - diger saticilar
      var otherSellers = extractOtherSellers();
      var specifications = extractSpecifications();

      // Buy Box analizi: en dusuk fiyatli saticiyi bul
      var buyBoxWinner = seller;
      var buyBoxPrice = price;
      var allSellers = [{
        name: seller || 'Aktif Satici',
        price: price,
        isBuyBox: true
      }];

      for (var i = 0; i < otherSellers.length; i++) {
        allSellers.push({
          name: otherSellers[i].name,
          price: otherSellers[i].price,
          isBuyBox: false
        });
      }

      // En dusuk fiyati bul
      var lowestPrice = price;
      var lowestSeller = seller || 'Aktif Satici';
      for (var j = 0; j < allSellers.length; j++) {
        if (allSellers[j].price > 0 && allSellers[j].price < lowestPrice) {
          lowestPrice = allSellers[j].price;
          lowestSeller = allSellers[j].name;
        }
      }

      return {
        platform: PLATFORM,
        title: title,
        price: price,
        originalPrice: parsePrice(getTextContent(originalPriceEl)),
        rating: rating,
        reviewCount: reviewCount,
        seller: seller,
        url: location.href,
        productId: productId,
        category: category,
        favoriteCount: 0,
        specifications: specifications,

        // Buy Box verileri (Hepsiburada'ya ozgu)
        buyBox: {
          currentWinner: buyBoxWinner || 'Bilinmiyor',
          currentPrice: buyBoxPrice,
          allSellers: allSellers,
          totalSellerCount: allSellers.length,
          lowestPrice: lowestPrice,
          lowestPriceSeller: lowestSeller,
          priceDifference: price > 0 && lowestPrice > 0
            ? Math.round((price - lowestPrice) * 100) / 100
            : 0
        }
      };
    } catch (err) {
      warn('Urun detay cikarma hatasi:', err);
      return null;
    }
  }

  // ─── Veri Cikarma: Magaza Sayfasi ──────────────────────────

  /**
   * Magaza sayfasindan satici ve urun bilgilerini cikarir.
   * @returns {Object|null}
   */
  function extractStoreData() {
    try {
      var storeNameEl = queryFirst(document, SELECTORS.storeName);
      var storeName = getTextContent(storeNameEl);

      var productCards = queryAllFirst(document, SELECTORS.storeProducts);
      var products = [];

      for (var i = 0; i < productCards.length; i++) {
        var product = extractProductFromCard(productCards[i]);
        if (product) {
          products.push(product);
        }
      }

      return {
        platform: PLATFORM,
        storeName: storeName,
        storeUrl: location.href,
        productCount: products.length,
        products: products
      };
    } catch (err) {
      warn('Magaza veri cikarma hatasi:', err);
      return null;
    }
  }

  // ─── Analiz Motoru ──────────────────────────────────────────

  /**
   * Cikarilan urun verisini PazarRadar motorlarina gonderir ve analiz sonucu dondurur.
   * @param {Object} data - Cikarilan urun verisi
   * @returns {Object} Zenginlestirilmis analiz sonucu
   */
  function analyzeProduct(data) {
    if (!data) return data;

    var result = {
      platform: PLATFORM,
      title: data.title,
      price: data.price,
      originalPrice: data.originalPrice,
      rating: data.rating,
      reviewCount: data.reviewCount,
      seller: data.seller,
      url: data.url,
      productId: data.productId,
      category: data.category,
      favoriteCount: data.favoriteCount || 0,
      profit: null,
      sales: null,
      comparison: null,
      categoryInfo: null,
      buyBox: data.buyBox || null,
      specifications: data.specifications || null
    };

    // Kategori bilgisi
    try {
      if (PR.CommissionDB && typeof PR.CommissionDB.findCategory === 'function') {
        result.categoryInfo = PR.CommissionDB.findCategory(PLATFORM, data.category);
      }
    } catch (err) {
      warn('Kategori arama hatasi:', err);
    }

    // Kar hesaplamasi
    try {
      if (PR.ProfitCalculator && typeof PR.ProfitCalculator.calculate === 'function') {
        result.profit = PR.ProfitCalculator.calculate({
          platform: PLATFORM,
          categoryKey: result.categoryInfo ? result.categoryInfo.key : '',
          categoryName: data.category,
          salePrice: data.price
        });
      }
    } catch (err) {
      warn('Kar hesaplama hatasi:', err);
    }

    // Satis tahmini
    try {
      if (PR.SalesEstimator && typeof PR.SalesEstimator.estimate === 'function') {
        result.sales = PR.SalesEstimator.estimate({
          platform: PLATFORM,
          categoryKey: result.categoryInfo ? result.categoryInfo.key : '',
          totalReviews: data.reviewCount,
          favoriteCount: data.favoriteCount || 0,
          currentPrice: data.price,
          averageRating: data.rating
        });
      }
    } catch (err) {
      warn('Satis tahmini hatasi:', err);
    }

    // Platform karsilastirmasi
    try {
      if (PR.ProfitCalculator && typeof PR.ProfitCalculator.compareAll === 'function') {
        result.comparison = PR.ProfitCalculator.compareAll({
          categoryName: data.category,
          categoryKey: result.categoryInfo ? result.categoryInfo.key : '',
          salePrice: data.price
        });
      }
    } catch (err) {
      warn('Platform karsilastirma hatasi:', err);
    }

    return result;
  }

  // ─── UI Enjeksiyonu ─────────────────────────────────────────

  /**
   * Arama sayfasina toplu analiz cubugu ve urun kartlarini enjekte eder.
   * @param {Object[]} products - Cikarilan urun listesi
   */
  function injectSearchUI(products) {
    try {
      if (!products || products.length === 0) return;
      if (!PR.InjectUI) {
        warn('InjectUI modulu yuklu degil');
        return;
      }

      // Onceki enjeksiyonlari temizle
      if (typeof PR.InjectUI.cleanup === 'function') {
        PR.InjectUI.cleanup();
      }

      // Analiz edilmis urunler
      var analyzedProducts = [];
      for (var i = 0; i < products.length; i++) {
        var analyzed = analyzeProduct(products[i]);
        if (analyzed) {
          analyzedProducts.push(analyzed);
        }
      }

      if (analyzedProducts.length === 0) return;

      // Toplam istatistikler
      var totalPrice = 0;
      var totalRating = 0;
      var ratedCount = 0;
      var totalReviews = 0;
      var avgProfit = 0;
      var profitCount = 0;

      for (var j = 0; j < analyzedProducts.length; j++) {
        var p = analyzedProducts[j];
        if (p.price > 0) totalPrice += p.price;
        if (p.rating > 0) {
          totalRating += p.rating;
          ratedCount++;
        }
        totalReviews += (p.reviewCount || 0);
        if (p.profit && p.profit.netMarginPercent != null) {
          avgProfit += p.profit.netMarginPercent;
          profitCount++;
        }
      }

      var aggregateData = {
        platform: PLATFORM,
        totalProducts: analyzedProducts.length,
        averagePrice: analyzedProducts.length > 0
          ? Math.round((totalPrice / analyzedProducts.length) * 100) / 100
          : 0,
        averageRating: ratedCount > 0
          ? Math.round((totalRating / ratedCount) * 10) / 10
          : 0,
        totalReviews: totalReviews,
        averageProfitMargin: profitCount > 0
          ? Math.round((avgProfit / profitCount) * 10) / 10
          : null,
        comparison: analyzedProducts[0] ? analyzedProducts[0].comparison : null
      };

      // Aggregate bar enjekte et
      if (typeof PR.InjectUI.createSearchAggregateBar === 'function') {
        PR.InjectUI.createSearchAggregateBar(aggregateData);
      }

      // Her urun kartina analiz karti enjekte et
      for (var k = 0; k < analyzedProducts.length; k++) {
        var product = analyzedProducts[k];
        var card = products[k].cardElement;

        if (!card || card.getAttribute(INJECTED_ATTR) === 'true') continue;

        card.setAttribute(INJECTED_ATTR, 'true');

        if (typeof PR.InjectUI.createSearchResultCard === 'function') {
          var uiCard = PR.InjectUI.createSearchResultCard(product);
          if (uiCard && card.parentElement) {
            card.parentElement.insertBefore(uiCard, card.nextSibling);
          }
        }
      }

      info('Arama UI enjekte edildi: ' + analyzedProducts.length + ' urun');
    } catch (err) {
      warn('Arama UI enjeksiyon hatasi:', err);
    }
  }

  // ─── Urun Skoru Hesaplama ─────────────────────────────────

  /**
   * Coklu faktorlu urun skoru hesaplar (0-10 arasi).
   * Faktorler: Rating (%25), Yorum hacmi (%20), Talep (%25), Trend (%15), Kar marji (%15)
   * @param {Object} params
   * @returns {number} 0-10 arasi skor
   */
  function calculateProductScore(params) {
    var scores = {};

    // 1. Rating skoru (%25 agirlik)
    var rating = Math.min(Math.max(params.rating || 0, 0), 5);
    if (rating === 0) {
      scores.rating = 5;
    } else if (rating >= 4.5) {
      scores.rating = 10;
    } else if (rating >= 4.0) {
      scores.rating = 8.5;
    } else if (rating >= 3.5) {
      scores.rating = 7;
    } else if (rating >= 3.0) {
      scores.rating = 5.5;
    } else if (rating >= 2.0) {
      scores.rating = 3.5;
    } else {
      scores.rating = 2;
    }

    // 2. Yorum hacmi skoru (%20 agirlik)
    var reviews = params.reviewCount || 0;
    if (reviews >= 1000) {
      scores.reviews = 10;
    } else if (reviews >= 500) {
      scores.reviews = 9;
    } else if (reviews >= 200) {
      scores.reviews = 8;
    } else if (reviews >= 100) {
      scores.reviews = 7;
    } else if (reviews >= 50) {
      scores.reviews = 6;
    } else if (reviews >= 20) {
      scores.reviews = 5;
    } else if (reviews >= 5) {
      scores.reviews = 3.5;
    } else {
      scores.reviews = 2;
    }

    // 3. Talep seviyesi skoru (%25 agirlik)
    var demandScores = {
      'very_high': 10, 'high': 8, 'medium': 6, 'low': 4, 'very_low': 2
    };
    scores.demand = demandScores[params.demandLevel] || 5;

    // 4. Trend skoru (%15 agirlik)
    var trendScores = { 'up': 9, 'stable': 6, 'down': 3 };
    scores.trend = trendScores[params.trend] || 6;

    // 5. Fiyat/Deger skoru (%15 agirlik)
    var commissionRate = params.commissionRate || 0;
    if (commissionRate <= 5) {
      scores.value = 9;
    } else if (commissionRate <= 10) {
      scores.value = 7.5;
    } else if (commissionRate <= 15) {
      scores.value = 6;
    } else if (commissionRate <= 25) {
      scores.value = 4.5;
    } else {
      scores.value = 3;
    }
    if (params.originalPrice > 0 && params.price > 0 && params.originalPrice > params.price) {
      var discountPct = ((params.originalPrice - params.price) / params.originalPrice) * 100;
      if (discountPct >= 10) {
        scores.value = Math.min(10, scores.value + 1);
      }
    }

    var weightedScore = (
      scores.rating * 0.25 +
      scores.reviews * 0.20 +
      scores.demand * 0.25 +
      scores.trend * 0.15 +
      scores.value * 0.15
    );

    // 0-5 araliginda sinirla (platformlarla uyumlu), 1 ondalik basamak
    var finalScore = Math.max(0, Math.min(10, weightedScore)) / 2;
    return Math.round(finalScore * 10) / 10;
  }

  /**
   * Ic analiz nesnesini createProductPanel'in beklediği duz formata donusturur.
   * @param {Object} analysis - analyzeProduct sonucu
   * @returns {Object} createProductPanel icin duz veri nesnesi
   */
  function flattenAnalysisForPanel(analysis) {
    var profit = analysis.profit || {};
    var sales = analysis.sales || {};

    // Coklu faktorlu urun skoru hesaplama (0-10)
    var productScore = calculateProductScore({
      rating: analysis.rating || 0,
      reviewCount: analysis.reviewCount || 0,
      favoriteCount: analysis.favoriteCount || 0,
      demandLevel: sales.demandLevel || 'very_low',
      trend: sales.trend || 'stable',
      estimatedMonthlySales: sales.estimatedMonthlySales || 0,
      price: analysis.price || 0,
      originalPrice: analysis.originalPrice || 0,
      commissionRate: profit.commissionRate || 0
    });

    return {
      productScore: productScore,
      estimatedMonthlySales: sales.estimatedMonthlySales || 0,
      estimatedMonthlyRevenue: sales.estimatedMonthlyRevenue || 0,
      reviewRate: analysis.reviewCount || 0,
      favoriteRate: analysis.favoriteCount || 0,
      trend: sales.trend || 'stable',
      salePrice: profit.salePrice || analysis.price || 0,
      commissionRate: profit.commissionRate || 0,
      commissionAmount: profit.commissionAmount || 0,
      commissionKDV: profit.commissionKDV || 0,
      platformFee: profit.platformFee || 0,
      platformFeeKDV: profit.platformFeeKDV || 0,
      cargoFee: profit.estimatedCargo || 0,
      netRevenue: profit.netRevenue || 0,
      comparison: analysis.comparison || null,
      priceHistory: null,
      isPro: userIsPro,
      trialDaysLeft: userTrialDaysLeft
    };
  }

  /**
   * Urun detay sayfasina analiz paneli enjekte eder.
   * @param {Object} productData - Cikarilan urun detay verisi
   */
  function injectProductUI(productData) {
    try {
      if (!productData) return;
      if (!PR.InjectUI) {
        warn('InjectUI modulu yuklu degil');
        return;
      }

      // Onceki enjeksiyonlari temizle
      if (typeof PR.InjectUI.cleanup === 'function') {
        PR.InjectUI.cleanup();
      }

      var analysis = analyzeProduct(productData);
      if (!analysis) return;

      // Ana urun paneli - duz formata cevir ve DOM'a ekle
      if (typeof PR.InjectUI.createProductPanel === 'function') {
        var panelData = flattenAnalysisForPanel(analysis);

        // Limit kontrolu - background'a sor, sonra paneli render et
        if (typeof chrome !== 'undefined' && chrome.runtime && chrome.runtime.sendMessage) {
          chrome.runtime.sendMessage({
            type: 'peek_usage',
            payload: { usageType: 'productAnalyses' }
          }, function (response) {
            if (chrome.runtime.lastError || !response) {
              renderHBPanel(panelData, analysis);
              return;
            }
            if (!response.allowed) {
              panelData.limitReached = true;
              panelData.limitInfo = response;
            }
            renderHBPanel(panelData, analysis);
          });
        } else {
          renderHBPanel(panelData, analysis);
        }
      }

      info('Urun detay UI enjekte edildi: ' + (analysis.title || ''));
    } catch (err) {
      warn('Urun detay UI enjeksiyon hatasi:', err);
    }
  }

  /**
   * Hepsiburada panel DOM'a ekleme yardimcisi.
   * @param {Object} panelData - Panel verisi
   * @param {Object} analysis - Analiz sonucu (buyBox icin)
   */
  function renderHBPanel(panelData, analysis) {
    try {
      var panelEl = PR.InjectUI.createProductPanel(panelData);
      if (panelEl) {
        var target = document.querySelector('.product-detail-wrapper')
          || document.querySelector('#product-detail-app')
          || document.querySelector('main')
          || document.body;
        target.appendChild(panelEl);
      }

      // Buy Box analiz paneli (Hepsiburada'ya ozgu) - limit yoksa goster
      if (!panelData.limitReached &&
          analysis.buyBox &&
          analysis.buyBox.totalSellerCount > 1 &&
          typeof PR.InjectUI.createBuyBoxPanel === 'function') {
        PR.InjectUI.createBuyBoxPanel(analysis.buyBox);
      }
    } catch (err) {
      warn('HB panel render hatasi:', err);
    }
  }

  /**
   * Magaza sayfasina magaza analiz paneli enjekte eder.
   * @param {Object} storeData - Cikarilan magaza verisi
   */
  function injectStoreUI(storeData) {
    try {
      if (!storeData) return;
      if (!PR.InjectUI) {
        warn('InjectUI modulu yuklu degil');
        return;
      }

      if (typeof PR.InjectUI.cleanup === 'function') {
        PR.InjectUI.cleanup();
      }

      // Magaza urunlerini analiz et
      var analyzedProducts = [];
      for (var i = 0; i < storeData.products.length; i++) {
        var analyzed = analyzeProduct(storeData.products[i]);
        if (analyzed) {
          analyzedProducts.push(analyzed);
        }
      }

      var storeAnalysis = {
        platform: PLATFORM,
        storeName: storeData.storeName,
        storeUrl: storeData.storeUrl,
        productCount: storeData.productCount,
        products: analyzedProducts
      };

      if (typeof PR.InjectUI.createStorePanel === 'function') {
        PR.InjectUI.createStorePanel(storeAnalysis);
      }

      info('Magaza UI enjekte edildi: ' + storeData.storeName);
    } catch (err) {
      warn('Magaza UI enjeksiyon hatasi:', err);
    }
  }

  // ─── Background Script Iletisimi ───────────────────────────

  /**
   * Cikarilan veriyi background script'e iletir.
   * @param {string} type - Mesaj tipi ('search' | 'product' | 'store')
   * @param {Object} data - Gonderilecek veri
   */
  function sendToBackground(type, data) {
    try {
      if (typeof chrome === 'undefined' || !chrome.runtime || !chrome.runtime.sendMessage) {
        return;
      }

      // cardElement gibi DOM referanslarini temizle (JSON serialize edilemez)
      var cleanData = JSON.parse(JSON.stringify(data, function (_key, value) {
        if (value instanceof HTMLElement) return undefined;
        return value;
      }));

      chrome.runtime.sendMessage({
        type: 'product_data',
        platform: PLATFORM,
        pageType: type,
        data: cleanData,
        url: location.href,
        timestamp: Date.now()
      }, function (_response) {
        // Callback hatalarini yut - background yanit vermese bile sorun degil
        if (chrome.runtime.lastError) {
          // Sessizce gec
        }
      });
    } catch (err) {
      warn('Background iletisim hatasi:', err);
    }
  }

  // ─── Sayfa Yoneticileri ─────────────────────────────────────

  /**
   * Arama sayfasi isleyicisi.
   */
  function handleSearch() {
    info('Arama sayfasi tespit edildi');

    var products = extractSearchProducts();
    if (products.length === 0) {
      // Icerik gecikmeyle yuklenebilir; yeniden dene
      setTimeout(function () {
        var retryProducts = extractSearchProducts();
        if (retryProducts.length > 0) {
          injectSearchUI(retryProducts);
          sendToBackground(PAGE_TYPES.SEARCH, retryProducts);
        }
      }, 1500);
      return;
    }

    injectSearchUI(products);
    sendToBackground(PAGE_TYPES.SEARCH, products);
  }

  /**
   * Urun detay sayfasi isleyicisi.
   */
  function handleProduct() {
    info('Urun detay sayfasi tespit edildi');

    var productData = extractProductDetail();
    if (productData) {
      injectProductUI(productData);
      sendToBackground(PAGE_TYPES.PRODUCT, productData);
      tryTrackProduct(productData);
      return;
    }

    // React/SPA render gecikmesi icin kademeli retry (500, 1500, 3000, 5000ms)
    var retryDelays = [200, 600, 1500, 3500];
    var retryIndex = 0;

    function retryExtract() {
      if (retryIndex >= retryDelays.length) {
        warn('Urun verisi ' + retryDelays.length + ' denemede alinamadi');
        return;
      }
      var delay = retryDelays[retryIndex];
      retryIndex++;
      setTimeout(function () {
        var retryData = extractProductDetail();
        if (retryData) {
          info('Urun verisi ' + retryIndex + '. denemede alindi');
          injectProductUI(retryData);
          sendToBackground(PAGE_TYPES.PRODUCT, retryData);
          tryTrackProduct(retryData);
        } else {
          retryExtract();
        }
      }, delay);
    }

    retryExtract();
  }

  /**
   * Magaza sayfasi isleyicisi.
   */
  function handleStore() {
    info('Magaza sayfasi tespit edildi');

    var storeData = extractStoreData();
    if (!storeData) {
      setTimeout(function () {
        var retryData = extractStoreData();
        if (retryData) {
          injectStoreUI(retryData);
          sendToBackground(PAGE_TYPES.STORE, retryData);
        }
      }, 1500);
      return;
    }

    injectStoreUI(storeData);
    sendToBackground(PAGE_TYPES.STORE, storeData);
  }

  /**
   * Urunu stok takibine almaya calisir.
   * @param {Object} productData
   */
  function tryTrackProduct(productData) {
    try {
      if (!productData || !productData.productId) return;
      if (!PR.SalesEstimator || typeof PR.SalesEstimator.trackProduct !== 'function') return;

      // Stok bilgisi sayfada gosteriliyorsa takip et
      // Hepsiburada genelde stok gostermez; bu alan ileride genisletilebilir
      PR.SalesEstimator.trackProduct(
        productData.productId,
        PLATFORM,
        0 // Stok bilgisi mevcut degilse 0
      );
    } catch (err) {
      warn('Stok takip hatasi:', err);
    }
  }

  // ─── SPA Navigasyon Yonetimi ───────────────────────────────

  /** Mevcut URL takibi */
  var currentUrl = location.href;

  /** Sayfa iceriginin degisimini izleyen observer */
  var observer = new MutationObserver(debounce(function () {
    try {
      if (location.href !== currentUrl) {
        currentUrl = location.href;
        info('SPA navigasyonu tespit edildi: ' + currentUrl);

        // Onceki enjeksiyonlari temizle
        if (PR.InjectUI && typeof PR.InjectUI.cleanup === 'function') {
          PR.InjectUI.cleanup();
        }

        // Yeni icerigin yuklenmesini bekle
        setTimeout(init, SPA_REINJECT_DELAY);
      }
    } catch (err) {
      warn('SPA observer hatasi:', err);
    }
  }, MUTATION_DEBOUNCE_MS));

  // ─── Baslatma ───────────────────────────────────────────────

  /**
   * Ana baslatma fonksiyonu. Sayfa tipini tespit edip ilgili isleyiciyi calistirir.
   */
  function init() {
    try {
      var pageType = detectPageType();
      info('Sayfa tipi: ' + pageType);

      switch (pageType) {
        case PAGE_TYPES.SEARCH:
          handleSearch();
          break;
        case PAGE_TYPES.PRODUCT:
          handleProduct();
          break;
        case PAGE_TYPES.STORE:
          handleStore();
          break;
        default:
          info('Desteklenmeyen sayfa tipi, islem yapilmiyor');
          break;
      }
    } catch (err) {
      warn('Baslatma hatasi:', err);
    }
  }

  // DOM hazir oldugundan emin ol
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  // SPA navigasyon degisikliklerini izle
  try {
    if (document.body) {
      observer.observe(document.body, { childList: true, subtree: true });
    } else {
      document.addEventListener('DOMContentLoaded', function () {
        observer.observe(document.body, { childList: true, subtree: true });
      });
    }
  } catch (err) {
    warn('MutationObserver baslatilamadi:', err);
  }

  info('Hepsiburada content script yuklendi');

})();
