/**
 * PazarRadar - Background Service Worker
 *
 * Chrome Extension Manifest V3 service worker. Mesaj yonlendirme,
 * veri depolama (IndexedDB + chrome.storage), lisans yonetimi,
 * periyodik alarm sistemi ve bildirim yonetimini saglar.
 *
 * @module background
 */
'use strict';

// =============================================================================
// Sabitler
// =============================================================================

/** @type {string} Log on-eki */
const LOG_PREFIX = '[PazarRadar BG]';

/** @type {string} IndexedDB veritabani adi */
const DB_NAME = 'PazarRadarDB';

/** @type {number} IndexedDB veritabani versiyonu */
const DB_VERSION = 1;

/** @type {number} Maksimum takip edilebilecek urun sayisi */
const MAX_TRACKED_PRODUCTS = 2000;

/** @type {number} Fiyat gecmisi saklama suresi (gun) */
const PRICE_HISTORY_RETENTION_DAYS = 180;

/** @type {number} Urun cache suresi (ms) - 1 saat */
const PRODUCT_CACHE_TTL = 3600000;

/** @type {RegExp} Lisans anahtari format dogrulamasi */
const LICENSE_KEY_PATTERN = /^PR-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$/;

/** @type {string} Backend API base URL */
const API_BASE_URL = 'https://pazar-radar.teknowebteknoloji.com/api';

/** @type {number} Grace period - sunucuya erisilemezse (7 gun) */
const GRACE_PERIOD_MS = 7 * 24 * 60 * 60 * 1000;

/** @type {string[]} Desteklenen pazaryeri alanlari */
const MARKETPLACE_DOMAINS = Object.freeze([
  'trendyol.com',
  'hepsiburada.com',
  'n11.com',
  'amazon.com.tr'
]);

/** @type {Record<string, {productAnalyses: number, searchPages: number, profitCalcs: number}>} Plan limitleri */
const PLAN_LIMITS = Object.freeze({
  free: Object.freeze({ productAnalyses: 5, searchPages: 3, profitCalcs: 10 }),
  trial: Object.freeze({ productAnalyses: Infinity, searchPages: Infinity, profitCalcs: Infinity }),
  pro: Object.freeze({ productAnalyses: Infinity, searchPages: Infinity, profitCalcs: Infinity }),
  business: Object.freeze({ productAnalyses: Infinity, searchPages: Infinity, profitCalcs: Infinity })
});

/** @type {object} Varsayilan kullanici ayarlari */
const DEFAULT_SETTINGS = Object.freeze({
  defaultCostPrice: null,
  defaultCargoWeight: 1,
  enableNotifications: true,
  enableSearchOverlay: true,
  enableProductPanel: true,
  theme: 'dark',
  language: 'tr'
});

/** @type {object} Varsayilan lisans bilgisi */
const DEFAULT_LICENSE = Object.freeze({
  key: null,
  plan: 'free',
  validUntil: null,
  email: null,
  activatedAt: null,
  lastValidated: null,
  trialEndsAt: null,
  trialActivated: false
});

// =============================================================================
// Yardimci Fonksiyonlar
// =============================================================================

/**
 * Bugunun tarih dizesini dondurur (YYYY-MM-DD).
 * @returns {string}
 */
function getTodayString() {
  return new Date().toISOString().split('T')[0];
}

/**
 * Verilen gun sayisi kadar onceki tarihin timestamp'ini dondurur.
 * @param {number} days - Gun sayisi
 * @returns {number} Timestamp (ms)
 */
function daysAgoTimestamp(days) {
  return Date.now() - (days * 24 * 60 * 60 * 1000);
}

/**
 * Hata mesajini loglar.
 * @param {string} context - Hata baglami
 * @param {Error|string} error - Hata nesnesi veya mesaji
 */
function logError(context, error) {
  const message = error instanceof Error ? error.message : String(error);
  const stack = error instanceof Error ? error.stack : '';
  // Service worker icinde console.error kullanimi kabul edilebilir
  // eslint-disable-next-line no-console
  console.error(`${LOG_PREFIX} ${context}:`, message, stack);
}

/**
 * Bilgi mesajini loglar.
 * @param {string} message - Log mesaji
 */
function logInfo(message) {
  // eslint-disable-next-line no-console
  console.info(`${LOG_PREFIX} ${message}`);
}

/**
 * Benzersiz cihaz kimligini dondurur.
 * Ilk cagrimda uretilir, sonraki cagrilarda storage'dan okunur.
 * @returns {Promise<string>} SHA-256 hash (hex)
 */
async function getDeviceId() {
  const { deviceId } = await chrome.storage.local.get('deviceId');
  if (deviceId) return deviceId;

  // Benzersiz veri toplama
  const raw = [
    navigator.userAgent,
    screen.width + 'x' + screen.height,
    screen.colorDepth,
    Intl.DateTimeFormat().resolvedOptions().timeZone,
    Date.now().toString(36),
    Math.random().toString(36)
  ].join('|');

  const encoder = new TextEncoder();
  const data = encoder.encode(raw);
  const hashBuffer = await crypto.subtle.digest('SHA-256', data);
  const hashArray = Array.from(new Uint8Array(hashBuffer));
  const hashHex = hashArray.map(b => b.toString(16).padStart(2, '0')).join('');

  await chrome.storage.local.set({ deviceId: hashHex });
  return hashHex;
}

/**
 * Lisans anahtarini sunucu API'si ile dogrular.
 * Sunucuya erisilemezse mevcut lisansi grace period icinde korur.
 *
 * @param {string} key - Lisans anahtari
 * @param {string} deviceId - Cihaz kimligi
 * @returns {Promise<{valid: boolean, plan?: string, validUntil?: number, error?: string}>}
 */
async function validateLicenseKeyRemote(key, deviceId) {
  // Format on kontrolu
  if (!key || !LICENSE_KEY_PATTERN.test(key.trim().toUpperCase())) {
    return { valid: false, error: 'Gecersiz lisans anahtari formati.' };
  }

  try {
    const response = await fetch(`${API_BASE_URL}/license/validate`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ key: key.trim().toUpperCase(), deviceId })
    });

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}`);
    }

    const result = await response.json();

    if (result.success && result.data) {
      return {
        valid: result.data.valid,
        plan: result.data.plan,
        validUntil: result.data.validUntil,
        status: result.data.status
      };
    }

    return { valid: false, error: 'Sunucu dogrulama basarisiz.' };
  } catch (err) {
    logError('validateLicenseKeyRemote', err);
    // Sunucuya erisilemedi - grace period kontrolu
    return { valid: false, error: 'network_error', offline: true };
  }
}

// =============================================================================
// IndexedDB Yardimci Sinifi
// =============================================================================

/**
 * PazarRadar IndexedDB wrapper sinifi.
 * Fiyat gecmisi ve urun cache islemleri icin kullanilir.
 */
class PazarRadarDB {
  constructor() {
    /** @type {IDBDatabase|null} */
    this.db = null;
  }

  /**
   * IndexedDB baglantisini acar. Zaten aciksa mevcut baglanti doner.
   * @returns {Promise<IDBDatabase>}
   */
  async open() {
    if (this.db) {
      return this.db;
    }

    return new Promise((resolve, reject) => {
      const request = indexedDB.open(DB_NAME, DB_VERSION);

      request.onupgradeneeded = (event) => {
        const db = event.target.result;

        // Fiyat gecmisi deposu
        if (!db.objectStoreNames.contains('priceHistory')) {
          const priceStore = db.createObjectStore('priceHistory', { keyPath: 'key' });
          priceStore.createIndex('platform', 'platform', { unique: false });
          priceStore.createIndex('lastUpdated', 'lastUpdated', { unique: false });
        }

        // Urun cache deposu
        if (!db.objectStoreNames.contains('productCache')) {
          const cacheStore = db.createObjectStore('productCache', { keyPath: 'key' });
          cacheStore.createIndex('cachedAt', 'cachedAt', { unique: false });
        }
      };

      request.onsuccess = (event) => {
        this.db = event.target.result;

        // Baglanti kapanirsa referansi temizle
        this.db.onclose = () => {
          this.db = null;
        };

        resolve(this.db);
      };

      request.onerror = (event) => {
        logError('IndexedDB open', event.target.error);
        reject(new Error('IndexedDB acilamadi'));
      };
    });
  }

  /**
   * Bir urune fiyat verisi ekler veya gunceller.
   * @param {string} platform - Platform adi (trendyol, hepsiburada, n11)
   * @param {string} productId - Urun ID
   * @param {object} priceData - Fiyat bilgisi
   * @param {number} priceData.price - Guncel fiyat
   * @param {number} [priceData.originalPrice] - Orijinal (indirim oncesi) fiyat
   * @param {number|null} [priceData.stock] - Stok miktari
   * @param {string} [priceData.title] - Urun adi
   * @param {string} [priceData.brand] - Marka
   * @param {string} [priceData.category] - Kategori
   * @returns {Promise<void>}
   */
  async addPriceEntry(platform, productId, priceData) {
    const db = await this.open();
    const key = `${platform}_${productId}`;
    const today = getTodayString();

    return new Promise((resolve, reject) => {
      const tx = db.transaction('priceHistory', 'readwrite');
      const store = tx.objectStore('priceHistory');
      const getRequest = store.get(key);

      getRequest.onsuccess = () => {
        let record = getRequest.result;

        if (record) {
          // Mevcut kaydi guncelle
          const existingEntryIndex = record.entries.findIndex(
            (entry) => entry.date === today
          );

          const newEntry = {
            date: today,
            price: priceData.price,
            originalPrice: priceData.originalPrice ?? null,
            stock: priceData.stock ?? null
          };

          if (existingEntryIndex >= 0) {
            // Bugun zaten bir kayit varsa uzerine yaz
            record = {
              ...record,
              entries: record.entries.map((entry, index) =>
                index === existingEntryIndex ? newEntry : entry
              ),
              title: priceData.title || record.title,
              brand: priceData.brand || record.brand,
              category: priceData.category || record.category,
              lastUpdated: Date.now()
            };
          } else {
            // Yeni gun kaydini ekle
            record = {
              ...record,
              entries: [...record.entries, newEntry],
              title: priceData.title || record.title,
              brand: priceData.brand || record.brand,
              category: priceData.category || record.category,
              lastUpdated: Date.now()
            };
          }

          store.put(record);
        } else {
          // Urun limiti kontrolu
          this._checkProductLimit(store).then((canAdd) => {
            if (!canAdd) {
              logInfo('Maksimum urun takip limitine ulasildi');
              resolve();
              return;
            }

            const newRecord = {
              key,
              productId,
              platform,
              title: priceData.title || '',
              brand: priceData.brand || '',
              category: priceData.category || '',
              entries: [
                {
                  date: today,
                  price: priceData.price,
                  originalPrice: priceData.originalPrice ?? null,
                  stock: priceData.stock ?? null
                }
              ],
              lastUpdated: Date.now()
            };

            store.put(newRecord);
          }).catch((err) => {
            logError('addPriceEntry limit check', err);
          });
        }
      };

      tx.oncomplete = () => resolve();
      tx.onerror = (event) => {
        logError('addPriceEntry tx', event.target.error);
        reject(new Error('Fiyat verisi eklenemedi'));
      };
    });
  }

  /**
   * Depodaki urun sayisinin limite ulasilip ulasilmadigini kontrol eder.
   * @param {IDBObjectStore} store - priceHistory object store
   * @returns {Promise<boolean>} Eklenebilir mi?
   * @private
   */
  async _checkProductLimit(store) {
    return new Promise((resolve) => {
      const countRequest = store.count();
      countRequest.onsuccess = () => {
        resolve(countRequest.result < MAX_TRACKED_PRODUCTS);
      };
      countRequest.onerror = () => {
        resolve(false);
      };
    });
  }

  /**
   * Bir urunun fiyat gecmisini dondurur.
   * @param {string} platform - Platform adi
   * @param {string} productId - Urun ID
   * @returns {Promise<object|null>} Fiyat gecmisi kaydi veya null
   */
  async getPriceHistory(platform, productId) {
    const db = await this.open();
    const key = `${platform}_${productId}`;

    return new Promise((resolve, reject) => {
      const tx = db.transaction('priceHistory', 'readonly');
      const store = tx.objectStore('priceHistory');
      const request = store.get(key);

      request.onsuccess = () => {
        resolve(request.result || null);
      };

      request.onerror = (event) => {
        logError('getPriceHistory', event.target.error);
        reject(new Error('Fiyat gecmisi alinamadi'));
      };
    });
  }

  /**
   * Urun verisini cache'e yazar.
   * @param {string} platform - Platform adi
   * @param {string} productId - Urun ID
   * @param {object} data - Cache'lenecek urun verisi
   * @returns {Promise<void>}
   */
  async cacheProduct(platform, productId, data) {
    const db = await this.open();
    const key = `${platform}_${productId}`;

    return new Promise((resolve, reject) => {
      const tx = db.transaction('productCache', 'readwrite');
      const store = tx.objectStore('productCache');

      store.put({
        key,
        platform,
        productId,
        data,
        cachedAt: Date.now(),
        ttl: PRODUCT_CACHE_TTL
      });

      tx.oncomplete = () => resolve();
      tx.onerror = (event) => {
        logError('cacheProduct', event.target.error);
        reject(new Error('Urun cache yazilamadi'));
      };
    });
  }

  /**
   * Cache'ten urun verisini okur. TTL gecmisse null doner.
   * @param {string} platform - Platform adi
   * @param {string} productId - Urun ID
   * @returns {Promise<object|null>} Cache'lenmis urun verisi veya null
   */
  async getCachedProduct(platform, productId) {
    const db = await this.open();
    const key = `${platform}_${productId}`;

    return new Promise((resolve, reject) => {
      const tx = db.transaction('productCache', 'readonly');
      const store = tx.objectStore('productCache');
      const request = store.get(key);

      request.onsuccess = () => {
        const record = request.result;
        if (!record) {
          resolve(null);
          return;
        }

        // TTL kontrolu
        const isExpired = (Date.now() - record.cachedAt) > record.ttl;
        resolve(isExpired ? null : record.data);
      };

      request.onerror = (event) => {
        logError('getCachedProduct', event.target.error);
        reject(new Error('Cache okunamadi'));
      };
    });
  }

  /**
   * Eski fiyat gecmisi kayitlarini ve suresi dolmus cache'i temizler.
   * @returns {Promise<{removedEntries: number, removedCache: number}>}
   */
  async cleanup() {
    const db = await this.open();
    const cutoffTimestamp = daysAgoTimestamp(PRICE_HISTORY_RETENTION_DAYS);
    const cutoffDate = new Date(cutoffTimestamp).toISOString().split('T')[0];
    let removedEntries = 0;
    let removedCache = 0;

    // Fiyat gecmisi temizligi
    await new Promise((resolve, reject) => {
      const tx = db.transaction('priceHistory', 'readwrite');
      const store = tx.objectStore('priceHistory');
      const cursorRequest = store.openCursor();

      cursorRequest.onsuccess = (event) => {
        const cursor = event.target.result;
        if (!cursor) {
          resolve();
          return;
        }

        const record = cursor.value;
        const filteredEntries = record.entries.filter(
          (entry) => entry.date >= cutoffDate
        );

        const trimmedCount = record.entries.length - filteredEntries.length;
        removedEntries += trimmedCount;

        if (filteredEntries.length === 0) {
          // Hic kayit kalmazsa urunu tamamen sil
          cursor.delete();
        } else if (trimmedCount > 0) {
          // Eski kayitlari cikarilmis halini yaz
          cursor.update({
            ...record,
            entries: filteredEntries,
            lastUpdated: record.lastUpdated
          });
        }

        cursor.continue();
      };

      cursorRequest.onerror = (event) => {
        logError('cleanup priceHistory', event.target.error);
        reject(new Error('Fiyat gecmisi temizlenemedi'));
      };
    });

    // Cache temizligi
    await new Promise((resolve, reject) => {
      const tx = db.transaction('productCache', 'readwrite');
      const store = tx.objectStore('productCache');
      const cursorRequest = store.openCursor();

      cursorRequest.onsuccess = (event) => {
        const cursor = event.target.result;
        if (!cursor) {
          resolve();
          return;
        }

        const record = cursor.value;
        const isExpired = (Date.now() - record.cachedAt) > record.ttl;

        if (isExpired) {
          cursor.delete();
          removedCache++;
        }

        cursor.continue();
      };

      cursorRequest.onerror = (event) => {
        logError('cleanup productCache', event.target.error);
        reject(new Error('Cache temizlenemedi'));
      };
    });

    logInfo(
      `Temizlik tamamlandi: ${removedEntries} eski fiyat kaydi, ${removedCache} suresi dolmus cache silindi`
    );

    return { removedEntries, removedCache };
  }

  /**
   * IndexedDB'deki istatistikleri toplar.
   * @returns {Promise<{totalProducts: number, totalEntries: number, oldestEntry: string|null, newestEntry: string|null}>}
   */
  async getStats() {
    const db = await this.open();

    return new Promise((resolve, reject) => {
      const tx = db.transaction('priceHistory', 'readonly');
      const store = tx.objectStore('priceHistory');
      const cursorRequest = store.openCursor();

      let totalProducts = 0;
      let totalEntries = 0;
      let oldestEntry = null;
      let newestEntry = null;

      cursorRequest.onsuccess = (event) => {
        const cursor = event.target.result;
        if (!cursor) {
          resolve({ totalProducts, totalEntries, oldestEntry, newestEntry });
          return;
        }

        const record = cursor.value;
        totalProducts++;
        totalEntries += record.entries.length;

        for (const entry of record.entries) {
          if (!oldestEntry || entry.date < oldestEntry) {
            oldestEntry = entry.date;
          }
          if (!newestEntry || entry.date > newestEntry) {
            newestEntry = entry.date;
          }
        }

        cursor.continue();
      };

      cursorRequest.onerror = (event) => {
        logError('getStats', event.target.error);
        reject(new Error('Istatistikler alinamadi'));
      };
    });
  }

  /**
   * Tum fiyat gecmisi verilerini dizi olarak dondurur (export icin).
   * @returns {Promise<Array<object>>}
   */
  async exportData() {
    const db = await this.open();

    return new Promise((resolve, reject) => {
      const tx = db.transaction('priceHistory', 'readonly');
      const store = tx.objectStore('priceHistory');
      const request = store.getAll();

      request.onsuccess = () => {
        resolve(request.result || []);
      };

      request.onerror = (event) => {
        logError('exportData', event.target.error);
        reject(new Error('Veri disari aktarilamadi'));
      };
    });
  }

  /**
   * Son guncellenen urunleri dondurur.
   * @param {number} [limit=20] - Maksimum kayit sayisi
   * @returns {Promise<Array<object>>}
   */
  async getRecentProducts(limit = 20) {
    const db = await this.open();

    return new Promise((resolve, reject) => {
      const tx = db.transaction('priceHistory', 'readonly');
      const store = tx.objectStore('priceHistory');
      const index = store.index('lastUpdated');
      const results = [];

      // Sondan basa dogru (en yeni en onde)
      const cursorRequest = index.openCursor(null, 'prev');

      cursorRequest.onsuccess = (event) => {
        const cursor = event.target.result;
        if (!cursor || results.length >= limit) {
          resolve(results);
          return;
        }

        const record = cursor.value;
        const latestEntry = record.entries[record.entries.length - 1] || null;

        results.push({
          key: record.key,
          productId: record.productId,
          platform: record.platform,
          title: record.title,
          brand: record.brand,
          category: record.category,
          latestPrice: latestEntry?.price ?? null,
          latestOriginalPrice: latestEntry?.originalPrice ?? null,
          entryCount: record.entries.length,
          lastUpdated: record.lastUpdated
        });

        cursor.continue();
      };

      cursorRequest.onerror = (event) => {
        logError('getRecentProducts', event.target.error);
        reject(new Error('Son urunler alinamadi'));
      };
    });
  }

  /**
   * Veritabani baglantisini kapatir.
   */
  close() {
    if (this.db) {
      this.db.close();
      this.db = null;
    }
  }
}

// =============================================================================
// Veritabani Tekil Ornegi
// =============================================================================

/** @type {PazarRadarDB} */
const db = new PazarRadarDB();

// =============================================================================
// Lisans Dogrulama
// =============================================================================

/**
 * Lisans anahtarini dogrular.
 * MVP surecinde format dogrulamasi yapar, uretimde sunucu API'si ile dogrulanir.
 *
 * @param {string} key - Lisans anahtari (PR-XXXX-XXXX-XXXX-XXXX formati)
 * @returns {{valid: boolean, plan?: string, validUntil?: number, error?: string}}
 */
function validateLicenseKey(key) {
  if (!key || typeof key !== 'string') {
    return { valid: false, error: 'Lisans anahtari bos olamaz' };
  }

  const trimmedKey = key.trim().toUpperCase();

  if (!LICENSE_KEY_PATTERN.test(trimmedKey)) {
    return { valid: false, error: 'Gecersiz lisans anahtari formati. Beklenen: PR-XXXX-XXXX-XXXX-XXXX' };
  }

  // MVP: Formati gecerli her anahtar PRO olarak kabul edilir
  // Uretimde: Sunucu API'si ile dogrulama yapilacak
  return {
    valid: true,
    plan: 'pro',
    validUntil: Date.now() + (365 * 24 * 60 * 60 * 1000) // 1 yil gecerli
  };
}

// =============================================================================
// Kullanim Limiti Kontrolu
// =============================================================================

/**
 * Kullanim limitini kontrol eder ve sayaci arttirir.
 * Yeni gun basinda sayaclari otomatik sifirlar.
 *
 * @param {'productAnalyses'|'searchPages'|'profitCalcs'} type - Kullanim tipi
 * @returns {Promise<{allowed: boolean, remaining: number, limit: number}>}
 */
async function checkUsageLimit(type) {
  const result = await chrome.storage.local.get(['license', 'usage']);
  const license = result.license || { ...DEFAULT_LICENSE };
  const plan = license.plan || 'free';
  const limits = PLAN_LIMITS[plan] || PLAN_LIMITS.free;
  const today = getTodayString();

  // Gunluk sifirlama
  let usage = result.usage || {};
  if (usage.date !== today) {
    usage = {
      date: today,
      productAnalyses: 0,
      searchPages: 0,
      profitCalcs: 0
    };
  }

  const currentCount = usage[type] || 0;
  const limit = limits[type];

  if (currentCount >= limit) {
    return { allowed: false, remaining: 0, limit };
  }

  // Sayaci artir ve kaydet
  const updatedUsage = {
    ...usage,
    [type]: currentCount + 1
  };
  await chrome.storage.local.set({ usage: updatedUsage });

  return {
    allowed: true,
    remaining: limit - (currentCount + 1),
    limit
  };
}

/**
 * Kullanim sayacini artirmadan sadece durumu kontrol eder.
 *
 * @param {'productAnalyses'|'searchPages'|'profitCalcs'} type - Kullanim tipi
 * @returns {Promise<{allowed: boolean, remaining: number, limit: number, current: number}>}
 */
async function peekUsageLimit(type) {
  const result = await chrome.storage.local.get(['license', 'usage']);
  const license = result.license || { ...DEFAULT_LICENSE };
  const plan = license.plan || 'free';
  const limits = PLAN_LIMITS[plan] || PLAN_LIMITS.free;
  const today = getTodayString();

  const usage = result.usage || {};
  const currentCount = (usage.date === today) ? (usage[type] || 0) : 0;
  const limit = limits[type];

  return {
    allowed: currentCount < limit,
    remaining: Math.max(0, limit - currentCount),
    limit,
    current: currentCount
  };
}

/**
 * peek_usage mesaj isleyicisi.
 * Sayaci artirmadan kullanim limitini kontrol eder.
 *
 * @param {object} payload
 * @param {'productAnalyses'|'searchPages'|'profitCalcs'} [payload.usageType] - Kullanim tipi
 * @returns {Promise<object>} Limit durumu
 */
async function handlePeekUsage(payload) {
  const usageType = payload.usageType || payload.type || 'productAnalyses';
  const result = await peekUsageLimit(usageType);
  return { success: true, ...result };
}

// =============================================================================
// Istatistik Yonetimi
// =============================================================================

/**
 * Global istatistik sayacini arttirir.
 * @param {'totalProductsAnalyzed'|'totalSearchPages'|'totalProfitCalcs'} key - Istatistik anahtari
 * @param {number} [amount=1] - Arttirilacak miktar
 * @returns {Promise<void>}
 */
async function incrementStat(key, amount = 1) {
  const { stats = {} } = await chrome.storage.local.get('stats');

  const updatedStats = {
    ...stats,
    [key]: (stats[key] || 0) + amount,
    lastActiveDate: Date.now()
  };

  await chrome.storage.local.set({ stats: updatedStats });
}

// =============================================================================
// CSV Export
// =============================================================================

/**
 * IndexedDB'deki fiyat gecmisi verilerini CSV formatina donusturur.
 * @returns {Promise<string>} CSV icerigi
 */
async function exportDataAsCSV() {
  const records = await db.exportData();

  const headers = [
    'Platform',
    'Urun ID',
    'Urun Adi',
    'Marka',
    'Kategori',
    'Tarih',
    'Fiyat',
    'Orijinal Fiyat',
    'Stok'
  ];

  const rows = [headers.join(',')];

  for (const record of records) {
    for (const entry of record.entries) {
      const row = [
        record.platform,
        record.productId,
        `"${(record.title || '').replace(/"/g, '""')}"`,
        `"${(record.brand || '').replace(/"/g, '""')}"`,
        `"${(record.category || '').replace(/"/g, '""')}"`,
        entry.date,
        entry.price ?? '',
        entry.originalPrice ?? '',
        entry.stock ?? ''
      ];
      rows.push(row.join(','));
    }
  }

  return rows.join('\n');
}

// =============================================================================
// Mesaj Isleyicileri
// =============================================================================

/**
 * Content script'ten gelen urun verisini isler.
 * Fiyat gecmisine ekler, cache'ler ve istatistikleri gunceller.
 *
 * @param {object} payload - Urun verisi
 * @returns {Promise<object>} Islem sonucu
 */
async function handleProductData(payload) {
  // Content script'ler { platform, data: {...} } formatinda gonderebilir
  const innerData = payload.data || {};
  const platform = payload.platform || innerData.platform;
  const productId = payload.productId || innerData.productId;
  const price = payload.price != null ? payload.price : innerData.price;
  const originalPrice = payload.originalPrice || innerData.originalPrice;
  const stock = payload.stock || innerData.stock;
  const title = payload.title || innerData.title;
  const brand = payload.brand || innerData.brand;
  const category = payload.category || innerData.category;
  const fullData = payload.fullData || innerData;

  if (!platform || !productId || price == null) {
    return { success: false, error: 'Eksik urun verisi: platform, productId ve price zorunlu' };
  }

  // Kullanim limitini kontrol et
  const usageCheck = await checkUsageLimit('productAnalyses');
  if (!usageCheck.allowed) {
    return {
      success: false,
      error: 'Gunluk urun analizi limitine ulasildi',
      usageInfo: usageCheck
    };
  }

  // Fiyat gecmisine ekle
  await db.addPriceEntry(platform, productId, {
    price,
    originalPrice,
    stock,
    title,
    brand,
    category
  });

  // Tam veriyi cache'le
  if (fullData) {
    await db.cacheProduct(platform, productId, fullData);
  }

  // Istatistik guncelle
  await incrementStat('totalProductsAnalyzed');

  return {
    success: true,
    usageInfo: usageCheck
  };
}

/**
 * Kullanim sayacini arttirir (genel amacli).
 *
 * @param {object} payload
 * @param {'productAnalyses'|'searchPages'|'profitCalcs'} payload.type - Kullanim tipi
 * @returns {Promise<object>} Limit durumu
 */
async function handleTrackUsage(payload) {
  const { type } = payload;

  const validTypes = ['productAnalyses', 'searchPages', 'profitCalcs'];
  if (!validTypes.includes(type)) {
    return { success: false, error: `Gecersiz kullanim tipi: ${type}` };
  }

  const usageCheck = await checkUsageLimit(type);

  // Istatistik eslestirmesi
  const statMap = {
    productAnalyses: 'totalProductsAnalyzed',
    searchPages: 'totalSearchPages',
    profitCalcs: 'totalProfitCalcs'
  };

  if (usageCheck.allowed) {
    await incrementStat(statMap[type]);
  }

  return {
    success: usageCheck.allowed,
    usageInfo: usageCheck,
    ...(usageCheck.allowed ? {} : { error: 'Gunluk kullanim limitine ulasildi' })
  };
}

/**
 * Bir urunun fiyat gecmisini dondurur.
 *
 * @param {object} payload
 * @param {string} payload.platform - Platform adi
 * @param {string} payload.productId - Urun ID
 * @returns {Promise<object>} Fiyat gecmisi
 */
async function handleGetPriceHistory(payload) {
  const { platform, productId } = payload;

  if (!platform || !productId) {
    return { success: false, error: 'platform ve productId zorunlu' };
  }

  const history = await db.getPriceHistory(platform, productId);

  if (!history) {
    return { success: true, data: null, message: 'Bu urun icin fiyat gecmisi bulunamadi' };
  }

  return { success: true, data: history };
}

/**
 * Kullanici ayarlarini ve lisans durumunu dondurur.
 *
 * @returns {Promise<object>} Ayarlar ve lisans bilgisi
 */
async function handleGetSettings() {
  const result = await chrome.storage.local.get(['settings', 'license']);
  const license = { ...DEFAULT_LICENSE, ...(result.license || {}) };

  // Grace period hesapla
  let gracePeriodInfo = null;
  if (license.key && license.lastValidated) {
    const gracePeriodEnd = license.lastValidated + GRACE_PERIOD_MS;
    const now = Date.now();
    if (now < gracePeriodEnd && (now - license.lastValidated) > 24 * 60 * 60 * 1000) {
      // Son 24 saatten uzun suredir dogrulama yapilamamissa grace period goster
      gracePeriodInfo = {
        active: true,
        daysLeft: Math.ceil((gracePeriodEnd - now) / (24 * 60 * 60 * 1000)),
        endsAt: gracePeriodEnd
      };
    }
  }

  return {
    success: true,
    data: {
      settings: { ...DEFAULT_SETTINGS, ...(result.settings || {}) },
      license: license,
      plan: license.plan || 'free',
      trialEndsAt: license.trialEndsAt || null,
      gracePeriod: gracePeriodInfo
    }
  };
}

/**
 * Dashboard verilerini toplar ve dondurur.
 *
 * @returns {Promise<object>} Dashboard verileri
 */
async function handleGetDashboardData() {
  const storageResult = await chrome.storage.local.get(['stats', 'usage', 'license']);
  const stats = storageResult.stats || {};
  const usage = storageResult.usage || {};
  const license = storageResult.license || { ...DEFAULT_LICENSE };
  const today = getTodayString();

  // Gunluk kullanimi sifirla (gerekirse)
  const todayUsage = (usage.date === today)
    ? usage
    : { date: today, productAnalyses: 0, searchPages: 0, profitCalcs: 0 };

  let dbStats = { totalProducts: 0, totalEntries: 0, oldestEntry: null, newestEntry: null };
  try {
    dbStats = await db.getStats();
  } catch (err) {
    logError('handleGetDashboardData dbStats', err);
  }

  return {
    success: true,
    data: {
      plan: license.plan || 'free',
      todayUsage,
      limits: PLAN_LIMITS[license.plan || 'free'],
      stats: {
        totalProductsAnalyzed: stats.totalProductsAnalyzed || 0,
        totalSearchPages: stats.totalSearchPages || 0,
        totalProfitCalcs: stats.totalProfitCalcs || 0,
        favoriteProducts: stats.favoriteProducts || [],
        installDate: stats.installDate || null,
        lastActiveDate: stats.lastActiveDate || null
      },
      db: dbStats
    }
  };
}

/**
 * Son analiz edilen urunleri dondurur.
 *
 * @param {object} payload
 * @param {number} [payload.limit=20] - Maksimum kayit sayisi
 * @returns {Promise<object>} Son urunler
 */
async function handleGetRecentAnalyses(payload) {
  const limit = payload?.limit || 20;
  const products = await db.getRecentProducts(limit);

  return {
    success: true,
    data: products
  };
}

/**
 * Kullanici ayarlarini kaydeder.
 *
 * @param {object} payload - Kaydedilecek ayarlar (kismi guncelleme destekler)
 * @returns {Promise<object>} Islem sonucu
 */
async function handleSaveSettings(payload) {
  if (!payload || typeof payload !== 'object') {
    return { success: false, error: 'Gecersiz ayar verisi' };
  }

  const { settings: currentSettings } = await chrome.storage.local.get('settings');
  const mergedSettings = {
    ...DEFAULT_SETTINGS,
    ...(currentSettings || {}),
    ...payload
  };

  // Gecersiz deger kontrolu
  const validThemes = ['dark', 'light'];
  const validLanguages = ['tr', 'en'];

  if (mergedSettings.theme && !validThemes.includes(mergedSettings.theme)) {
    return { success: false, error: `Gecersiz tema: ${mergedSettings.theme}` };
  }

  if (mergedSettings.language && !validLanguages.includes(mergedSettings.language)) {
    return { success: false, error: `Gecersiz dil: ${mergedSettings.language}` };
  }

  await chrome.storage.local.set({ settings: mergedSettings });

  return { success: true, data: mergedSettings };
}

/**
 * Lisans anahtarini dogrular (sunucu dogrulamasi simule eder).
 *
 * @returns {Promise<object>} Lisans durumu
 */
async function handleCheckLicense() {
  const { license } = await chrome.storage.local.get('license');

  if (!license || !license.key) {
    return {
      success: true,
      data: { ...DEFAULT_LICENSE, plan: 'free' }
    };
  }

  // Surenin dolup dolmadigini kontrol et
  if (license.validUntil && Date.now() > license.validUntil) {
    const expiredLicense = {
      ...license,
      plan: 'free',
      lastValidated: Date.now()
    };
    await chrome.storage.local.set({ license: expiredLicense });

    return {
      success: true,
      data: expiredLicense,
      message: 'Lisans suresi dolmus, ucretsiz plana gecildi'
    };
  }

  // Dogrulama tarihini guncelle
  const validatedLicense = {
    ...license,
    lastValidated: Date.now()
  };
  await chrome.storage.local.set({ license: validatedLicense });

  return { success: true, data: validatedLicense };
}

/**
 * Yeni bir lisans anahtarini aktif eder.
 * Once sunucu API'sine aktivasyon istegi gonderir.
 * Sunucuya erisilemezse fallback olarak lokal dogrulama kullanir.
 *
 * @param {object} payload
 * @param {string} payload.key - Lisans anahtari
 * @param {string} [payload.email] - Kullanici e-posta adresi
 * @returns {Promise<object>} Aktivasyon sonucu
 */
async function handleActivateLicense(payload) {
  const { key, email } = payload;

  // Format on kontrolu
  const localValidation = validateLicenseKey(key);
  if (!localValidation.valid) {
    return { success: false, error: localValidation.error };
  }

  const trimmedKey = key.trim().toUpperCase();
  const deviceId = await getDeviceId();

  // Mevcut license'dan trialEndsAt degerini koru
  const { license: existingLicense } = await chrome.storage.local.get('license');
  const preservedTrialEndsAt = existingLicense?.trialEndsAt || null;

  // Sunucu API'sine aktivasyon istegi
  try {
    const response = await fetch(`${API_BASE_URL}/license/activate`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ key: trimmedKey, deviceId, email })
    });

    const result = await response.json();

    if (result.success) {
      const license = {
        key: trimmedKey,
        plan: result.data.plan,
        validUntil: result.data.validUntil,
        email: email || null,
        activatedAt: Date.now(),
        lastValidated: Date.now(),
        trialEndsAt: preservedTrialEndsAt
      };

      await chrome.storage.local.set({ license });

      return {
        success: true,
        data: license,
        message: `${result.data.plan.toUpperCase()} plan basariyla aktif edildi`
      };
    }

    // Sunucu hata dondurduyse
    return {
      success: false,
      error: result.error?.message || 'Lisans aktive edilemedi.'
    };
  } catch (err) {
    logError('handleActivateLicense:remote', err);

    // Sunucuya erisilemedi - lokal fallback (MVP uyumluluk)
    const license = {
      key: trimmedKey,
      plan: localValidation.plan,
      validUntil: localValidation.validUntil,
      email: email || null,
      activatedAt: Date.now(),
      lastValidated: Date.now(),
      trialEndsAt: preservedTrialEndsAt
    };

    await chrome.storage.local.set({ license });

    return {
      success: true,
      data: license,
      message: `${localValidation.plan.toUpperCase()} plan aktif edildi (cevrimdisi mod)`
    };
  }
}

/**
 * Kullanim istatistiklerini dondurur.
 *
 * @returns {Promise<object>} Kullanim istatistikleri
 */
async function handleGetUsageStats() {
  const result = await chrome.storage.local.get(['usage', 'license', 'stats']);
  const license = result.license || { ...DEFAULT_LICENSE };
  const usage = result.usage || {};
  const stats = result.stats || {};
  const today = getTodayString();
  const plan = license.plan || 'free';

  const todayUsage = (usage.date === today)
    ? usage
    : { date: today, productAnalyses: 0, searchPages: 0, profitCalcs: 0 };

  const limits = PLAN_LIMITS[plan];

  return {
    success: true,
    data: {
      plan,
      today: todayUsage,
      limits,
      remaining: {
        productAnalyses: Math.max(0, limits.productAnalyses - (todayUsage.productAnalyses || 0)),
        searchPages: Math.max(0, limits.searchPages - (todayUsage.searchPages || 0)),
        profitCalcs: Math.max(0, limits.profitCalcs - (todayUsage.profitCalcs || 0))
      },
      allTime: {
        totalProductsAnalyzed: stats.totalProductsAnalyzed || 0,
        totalSearchPages: stats.totalSearchPages || 0,
        totalProfitCalcs: stats.totalProfitCalcs || 0,
        installDate: stats.installDate || null,
        lastActiveDate: stats.lastActiveDate || null
      }
    }
  };
}

/**
 * Takip edilen urunleri CSV olarak disari aktarir.
 *
 * @returns {Promise<object>} CSV verisi
 */
async function handleExportData() {
  const csv = await exportDataAsCSV();

  return {
    success: true,
    data: {
      csv,
      filename: `pazarradar-export-${getTodayString()}.csv`,
      mimeType: 'text/csv;charset=utf-8;'
    }
  };
}

// =============================================================================
// Mesaj Router
// =============================================================================

/** @type {Record<string, function>} Mesaj tipi -> isleyici eslestirmesi */
const MESSAGE_HANDLERS = Object.freeze({
  // Content script mesajlari
  product_data: handleProductData,
  track_usage: handleTrackUsage,
  get_price_history: handleGetPriceHistory,
  get_settings: handleGetSettings,
  peek_usage: handlePeekUsage,

  // Popup / side panel mesajlari
  get_dashboard_data: handleGetDashboardData,
  get_recent_analyses: handleGetRecentAnalyses,
  save_settings: handleSaveSettings,
  check_license: handleCheckLicense,
  activate_license: handleActivateLicense,
  get_usage_stats: handleGetUsageStats,
  export_data: handleExportData
});

/**
 * Chrome runtime mesaj dinleyicisi.
 * Gelen mesajlari uygun isleyiciye yonlendirir.
 */
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  const { type, action, payload, ...rest } = message || {};
  const msgType = type || action;

  if (!msgType) {
    sendResponse({ success: false, error: 'Mesaj tipi belirtilmedi' });
    return false;
  }

  const handler = MESSAGE_HANDLERS[msgType];

  if (!handler) {
    sendResponse({ success: false, error: `Bilinmeyen mesaj tipi: ${msgType}` });
    return false;
  }

  // Content script'ler payload'u farkli formatta gonderebilir - normalize et
  const normalizedPayload = payload || rest || {};

  // Asenkron isleyiciyi calistir
  (async () => {
    try {
      const result = await handler(normalizedPayload);
      sendResponse(result);
    } catch (error) {
      logError(`handler:${msgType}`, error);
      sendResponse({
        success: false,
        error: `Islem sirasinda hata olustu: ${error.message}`
      });
    }
  })();

  // true dondurerek asenkron sendResponse kullanacagimizi belirtiyoruz
  return true;
});

// =============================================================================
// Alarm Sistemi
// =============================================================================

/**
 * Alarm olay isleyicisi.
 * Periyodik gorevleri zamaninda calistirir.
 */
chrome.alarms.onAlarm.addListener(async (alarm) => {
  try {
    switch (alarm.name) {
      case 'daily_reset': {
        // Saatlik kontrol: gunluk sayaclarin sifirlanmasi gerekiyor mu?
        const { usage } = await chrome.storage.local.get('usage');
        const today = getTodayString();

        if (usage && usage.date !== today) {
          await chrome.storage.local.set({
            usage: {
              date: today,
              productAnalyses: 0,
              searchPages: 0,
              profitCalcs: 0
            }
          });
          logInfo('Gunluk kullanim sayaclari sifirlandi');
        }
        break;
      }

      case 'weekly_cleanup': {
        logInfo('Haftalik temizlik baslatiliyor...');
        const result = await db.cleanup();
        logInfo(
          `Temizlik tamamlandi: ${result.removedEntries} fiyat kaydi, ${result.removedCache} cache silindi`
        );
        break;
      }

      case 'license_check': {
        // Gunluk lisans dogrulamasi
        const { license } = await chrome.storage.local.get('license');

        // Trial suresi kontrolu
        if (license && license.plan === 'trial') {
          if (license.trialEndsAt && Date.now() > license.trialEndsAt) {
            await chrome.storage.local.set({
              license: { ...license, plan: 'free', lastValidated: Date.now() }
            });
            logInfo('Trial suresi doldu, ucretsiz plana gecildi');

            const { settings } = await chrome.storage.local.get('settings');
            if (settings?.enableNotifications) {
              chrome.notifications.create('trial_expired', {
                type: 'basic',
                iconUrl: 'icons/icon128.png',
                title: 'PazarRadar - Deneme Suresi Doldu',
                message: '14 gunluk PRO denemeniz sona erdi. PRO\'ya yukselterek tam erisim elde edin.'
              });
            }
          }
          break;
        }

        if (!license || !license.key) {
          break;
        }

        // Sunucu ile dogrulama yap
        const deviceId = await getDeviceId();
        const remoteResult = await validateLicenseKeyRemote(license.key, deviceId);

        if (remoteResult.offline) {
          // Sunucuya erisilemedi - grace period kontrolu
          const lastValidated = license.lastValidated || 0;
          const gracePeriodEnd = lastValidated + GRACE_PERIOD_MS;

          if (Date.now() > gracePeriodEnd) {
            // Grace period doldu
            await chrome.storage.local.set({
              license: { ...license, plan: 'free', lastValidated: Date.now() }
            });
            logInfo('Grace period doldu, sunucuya erisilemedi. Ucretsiz plana gecildi.');

            const { settings: s } = await chrome.storage.local.get('settings');
            if (s?.enableNotifications) {
              chrome.notifications.create('license_grace_expired', {
                type: 'basic',
                iconUrl: 'icons/icon128.png',
                title: 'PazarRadar - Dogrulama Basarisiz',
                message: 'Lisansiniz dogrulanamadi. Ucretsiz plana gecildi.'
              });
            }
          } else {
            logInfo(`Sunucuya erisilemedi. Grace period aktif (${Math.ceil((gracePeriodEnd - Date.now()) / 86400000)} gun kaldi).`);
          }
          break;
        }

        // Sunucudan cevap geldi
        if (remoteResult.valid) {
          await chrome.storage.local.set({
            license: {
              ...license,
              plan: remoteResult.plan || license.plan,
              validUntil: remoteResult.validUntil || license.validUntil,
              lastValidated: Date.now()
            }
          });
          logInfo(`Lisans dogrulandi: ${remoteResult.plan}`);
        } else {
          // Gecersiz/expired
          const expiredLicense = {
            ...license,
            plan: 'free',
            lastValidated: Date.now()
          };
          await chrome.storage.local.set({ license: expiredLicense });
          logInfo(`Lisans gecersiz: ${remoteResult.status || 'unknown'}. Ucretsiz plana gecildi.`);

          const { settings } = await chrome.storage.local.get('settings');
          if (settings?.enableNotifications) {
            chrome.notifications.create('license_expired', {
              type: 'basic',
              iconUrl: 'icons/icon128.png',
              title: 'PazarRadar - Lisans Suresi Doldu',
              message: 'PRO lisansiniz gecersiz. Ucretsiz plana gecildi.'
            });
          }
        }
        break;
      }

      default:
        logInfo(`Bilinmeyen alarm: ${alarm.name}`);
    }
  } catch (error) {
    logError(`alarm:${alarm.name}`, error);
  }
});

// =============================================================================
// Side Panel Yonetimi
// =============================================================================

/**
 * Sayfa guncelleme olaylarini dinleyerek pazaryeri sayfalarinda
 * side panel'i aktif hale getirir.
 */
chrome.tabs.onUpdated.addListener(async (tabId, changeInfo, tab) => {
  if (changeInfo.status !== 'complete') {
    return;
  }

  try {
    const url = tab.url || '';
    const isMarketplace = MARKETPLACE_DOMAINS.some(
      (domain) => url.includes(domain)
    );

    await chrome.sidePanel.setOptions({
      tabId,
      path: 'sidepanel/panel.html',
      enabled: isMarketplace
    });
  } catch (error) {
    // Tab kapanmis veya erisim yoksa sessizce atla
    logError('sidePanel.setOptions', error);
  }
});

// =============================================================================
// Kurulum ve Baslangic
// =============================================================================

/**
 * Eklenti ilk kurulumunda veya guncellemesinde calisir.
 * Varsayilan ayarlari olusturur ve alarm zamanlayicilarini baslatir.
 */
chrome.runtime.onInstalled.addListener(async (details) => {
  try {
    if (details.reason === 'install') {
      // Ilk kurulum: varsayilan degerleri ata + 14 gun trial baslat
      var trialDays = 14;
      var trialEndsAt = Date.now() + (trialDays * 24 * 60 * 60 * 1000);
      await chrome.storage.local.set({
        settings: { ...DEFAULT_SETTINGS },
        license: {
          ...DEFAULT_LICENSE,
          plan: 'trial',
          trialEndsAt: trialEndsAt,
          trialActivated: true
        },
        usage: {
          date: getTodayString(),
          productAnalyses: 0,
          searchPages: 0,
          profitCalcs: 0
        },
        stats: {
          totalProductsAnalyzed: 0,
          totalSearchPages: 0,
          totalProfitCalcs: 0,
          favoriteProducts: [],
          installDate: Date.now(),
          lastActiveDate: Date.now()
        }
      });

      logInfo('Ilk kurulum tamamlandi, varsayilan ayarlar olusturuldu');
    }

    if (details.reason === 'update') {
      // Guncelleme: mevcut ayarlari varsayilanlarla birlestirilmis halde kaydet
      // (yeni ayar anahtarlari icin)
      const { settings, stats } = await chrome.storage.local.get(['settings', 'stats']);

      await chrome.storage.local.set({
        settings: { ...DEFAULT_SETTINGS, ...(settings || {}) },
        stats: {
          totalProductsAnalyzed: 0,
          totalSearchPages: 0,
          totalProfitCalcs: 0,
          favoriteProducts: [],
          installDate: null,
          lastActiveDate: null,
          ...(stats || {})
        }
      });

      logInfo(`Eklenti guncellendi: v${details.previousVersion || '?'} -> mevcut`);
    }

    // Side panel davranisini ayarla
    await chrome.sidePanel.setPanelBehavior({ openPanelOnActionClick: false });

    // Alarmlari olustur (zaten varsa yeniden olusturulur)
    await chrome.alarms.create('daily_reset', { periodInMinutes: 60 });
    await chrome.alarms.create('weekly_cleanup', { periodInMinutes: 10080 });
    await chrome.alarms.create('license_check', { periodInMinutes: 1440 });

    logInfo('Alarm zamanlayicilari olusturuldu');
  } catch (error) {
    logError('onInstalled', error);
  }
});

/**
 * Service worker basladiginda IndexedDB'yi acarak hazir olmasini saglar.
 */
(async () => {
  try {
    await db.open();
    logInfo('Service worker baslatildi, IndexedDB hazir');
  } catch (error) {
    logError('startup', error);
  }
})();
